#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from zope.annotation import IAttributeAnnotatable
from zope.interface import Attribute, Interface, Invalid, invariant
from zope.schema import Bool, Choice, Int, Object, Set, Text, TextLine, Tuple

from onf_website.reference.forest.schema import ForestsListField
from onf_website.reference.insee.schema import CommunesListField
from onf_website.reference.orga.schema import StructuresListField
from pyams_content.shared.view import IViewSettings
from pyams_gis.interfaces import IGeoPoint
from pyams_gis.schema import GeoAreaField, GeoPointField
from pyams_utils.interfaces.inherit import IInheritInfo
from pyams_utils.schema import PersistentDict


__docformat__ = 'restructuredtext'

from onf_website import _


#
# Location manager settings
#

LOCATION_SETTINGS = {
    'gps': _("GPS position"),
    'location': _("Location (forest, city, address)"),
    'insee': _("Administrative level (country, department)"),
    'structure': _("ONF organisation"),
    'label': _("Alternate label")
}
LOCATION_SETTINGS_KEYS = ('gps', 'location', 'insee', 'structure', 'label')

LOCATION_SETTINGS_KEY = 'onf_website.location.settings'


class ILocationSetting(Interface):
    """Location setting state interface"""

    name = Attribute("Setting name")

    visible = Bool(title=_("Visible?"),
                   required=True,
                   default=True)

    unfolded = Bool(title=_("Unfolded?"),
                    required=True,
                    default=True)


class ILocationManagerSettings(Interface):
    """Location manager settings

    This interface is used to define which location components are available,
    and if they are enabled and displayed by default for a given content.
    """

    settings = PersistentDict(title=_("Location settings"),
                              key_type=Choice(values=LOCATION_SETTINGS_KEYS),
                              value_type=Object(schema=ILocationSetting))


class ILocationManagerTarget(IAttributeAnnotatable):
    """Location manager target marker interface"""


#
# Content location interfaces
#

LOCATION_INFO_KEY = 'onf_website.location'


class ILocationGPSInfo(Interface):
    """GPS coordinates location interface"""

    gps_location = GeoPointField(title=_("Geographic position"),
                                 required=False)


class ILocationLocationInfo(Interface):
    """Location base location interface"""

    forests = ForestsListField(title=_("Managed forests"),
                               description=_("Assisted input allows you to select one or more "
                                             "forest(s) managed by the National forestry office"),
                               required=False)

    def get_forests_list(self, params=None):
        """Get list of forests with national ID and label"""

    all_departments_forests = Bool(title=_("All forests of selected departments"),
                                   required=False,
                                   default=False)

    domanials_only = Bool(title=_("All domanial forests"),
                          required=False,
                          default=False)

    communals_only = Bool(title=_("All communal forests"),
                          required=False,
                          default=False)

    get_forests_from_departments = Attribute("Boolean flag used to specified if forests list "
                                             "is extracted from selected departments")

    forests_index = Attribute("Forests list used for indexing")

    @invariant
    def check_forests(self):
        if len(tuple(filter(bool, (self.all_departments_forests,
                                   self.domanials_only,
                                   self.communals_only)))) > 1:
            raise Invalid(_("You can choose all forests, domanial forests only or "
                            "communal forests only, but not several kinds of them"))

    cities = CommunesListField(title=_("Cities"),
                               required=False)

    address = Text(title=_("Address"),
                   required=False)

    situation = TextLine(title=_("Relative situation"),
                         description=_("For example: 25km north-west from Paris"),
                         required=False)

    natural_region = TextLine(title=_("Natural region"),
                              required=False)


class ILocationInseeInfo(Interface):
    """Location INSEE interface"""

    departments = Set(title=_("Departments"),
                      description=_("WARNING: if forests selection is based on departments "
                                    "selection, departments selected here will not be usable to "
                                    "locate this content, but only to find matching location "
                                    "forests"),
                      value_type=Choice(vocabulary='ONF regions with departments'),
                      required=False)

    def get_departments(self, not_if_forests=True):
        """Get full list of selected departments

        :param not_if_forests: if argument is True (default), and if forests list is extracted
          from selected departments, selected departments list is not used for indexing but
          only to get list of matching forests
        """

    selected_departments_index = Attribute("Selected departments list")

    departments_index = Attribute("Departments list used for indexing")

    def get_region_and_departments(self):
        """Get label of location departments and regions in a mapping"""

    countries = Set(title=_("Foreign countries"),
                    value_type=Choice(vocabulary='ONF countries'),
                    required=False)

    countries_index = Attribute("Countries list used for indexing")


class ILocationStructureInfo(Interface):
    """Location structures interface"""

    structures = StructuresListField(title=_("Structures"),
                                     required=False)


class ILocationLabelInfo(Interface):
    """Location label interface"""

    automatic_label = TextLine(title=_("Automatic label"),
                               description=_("This label is created automatically from above "
                                             "settings"),
                               required=False,
                               readonly=True)

    public_label = TextLine(title=_("Public label"),
                            description=_("Enter custom label if automatic label is not "
                                          "suitable"),
                            required=False)

    def get_elements(self):
        """Get internal elements matching selected location"""


class ILocationInfo(IInheritInfo, ILocationGPSInfo, ILocationLocationInfo, ILocationInseeInfo,
                    ILocationStructureInfo, ILocationLabelInfo, IAttributeAnnotatable):
    """Location information interface"""

    locations = Tuple(title=_("GPS locations"),
                      description=_("List of GPS coordinates associated with this location"),
                      value_type=Object(schema=IGeoPoint),
                      required=False)

    def get_gps_locations(self):
        """Get list of GPS locations"""

    def update_gps_locations(self):
        """Update stored GPS locations"""


class ILocationTarget(IAttributeAnnotatable):
    """Location marker interface"""


VIEW_LOCATION_SETTINGS_KEY = 'onf_website.view.location'


class IViewLocationSettings(IViewSettings):
    """View location settings"""

    select_context_gps_location = Bool(title=_("Select context GPS location?"),
                                       description=_("If 'yes', GPS coordinates will be "
                                                     "extracted from context location"),
                                       required=True,
                                       default=False)

    gps_location = GeoPointField(title=_("Other geographic position"),
                                 required=False)

    gps_distance = Int(title=_("Distance from GPS position"),
                       description=_("Distance from GPS location, given in kilometers; "
                                     "requires ElasticSearch extension"),
                       required=False)

    gps_area = GeoAreaField(title=_("GPS area rect"),
                            description=_("Area from which items should be selected; requires "
                                          "ElasticSearch extension"),
                            required=False)

    def get_gps_location(self, context):
        """Get GPS area for given context"""

    select_context_forests = Bool(title=_("Select context forests?"),
                                  description=_("If 'yes', forests will be extracted from "
                                                "context location"),
                                  required=True,
                                  default=False)

    forests = ForestsListField(title=_("Other forests"),
                               required=False)

    def get_forests(self, context):
        """Get all forests for given context"""

    select_context_cities = Bool(title=_("Select context cities?"),
                                 description=_("If 'yes', cities will be extracted from context "
                                               "location"),
                                 required=True,
                                 default=False)

    cities = CommunesListField(title=_("Other cities"),
                               required=False)

    def get_cities(self, context):
        """Get all cities for given context"""

    select_context_departments = Bool(title=_("Select context departments?"),
                                      description=_("If 'yes', departments will be extracted "
                                                    "from context location"),
                                      required=True,
                                      default=False)

    departments = Set(title=_("Other departments"),
                      value_type=Choice(vocabulary='ONF regions with departments'),
                      required=False)

    def get_departments(self, context):
        """Get all departments for given context"""

    def get_departments_index(self, context):
        """Get all departments index values for given context"""

    select_context_countries = Bool(title=_("Select context countries?"),
                                    description=_("If 'yes', countries will be extracted from "
                                                  "context location"),
                                    required=True,
                                    default=False)

    countries = Set(title=_("Other countries"),
                    value_type=Choice(vocabulary='ONF countries'),
                    required=False)

    def get_countries(self, context):
        """Get all countries for given context"""

    def get_countries_index(self, context):
        """Get all countries index values for given context"""

    select_context_structures = Bool(title=_("Select context structures?"),
                                     description=_("If 'yes', structures will be extracted from "
                                                   "context location"),
                                     required=True,
                                     default=False)

    structures = StructuresListField(title=_("Other structures"),
                                     required=False)

    def get_structures(self, context):
        """Get all structures for given context"""
