#
# Copyright (c) 2015-2023 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS_*** module

"""

from colander import Boolean, Float, MappingSchema, SchemaNode, SequenceSchema, String, drop

from pyams_utils.rest import StringListSchema

__docformat__ = 'restructuredtext'


class GPSCoordinates(MappingSchema):
    """GPS coordinates"""
    longitude = SchemaNode(Float(),
                           description="Longitude, in WGS84 coordinates")
    latitude = SchemaNode(Float(),
                          description="Latitude, in WGS84 coordinates")


class ForestIDsList(StringListSchema):
    """Forest IDs list"""


class ForestInfo(MappingSchema):
    """Forest information"""
    id_nat_frt = SchemaNode(String(),
                            description="National forest ID")
    libelle_usage = SchemaNode(String(),
                               description="Forest usage label")
    libelle_strict = SchemaNode(String(),
                                description="Forest strict label")


class ForestsList(SequenceSchema):
    """Forests list"""
    forest = ForestInfo(description="Forests list")


class CitiesIDsList(StringListSchema):
    """Cities INSEE codes list"""


class DepartmentsList(StringListSchema):
    """Departments INSEE codes list"""


class Location(MappingSchema):
    """Location schema"""
    gps_location = GPSCoordinates(description="GPS location",
                                  missing=drop)
    forests = ForestsList(description="Forests list",
                          missing=drop)
    cities = CitiesIDsList(description="Cities INSEE codes list",
                           missing=drop)
    departments = DepartmentsList(description="Departments INSEE codes list",
                                  missing=drop)
    all_departments_forests = SchemaNode(Boolean(),
                                         description="If true, forests list is extracted from departments list",
                                         missing=drop)
    domanials_only = SchemaNode(Boolean(),
                                description="If true, only domanial forests are extracted from departments list",
                                missing=drop)
    communals_only = SchemaNode(Boolean(),
                                description="If true, only communal forests are extracted from departments list",
                                missing=drop)


class LocationSetter(Location):
    """Location setter schema"""
    forests = ForestIDsList(description="Forests IDs list",
                            missing=drop)
