#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from elasticsearch_dsl import Q
from hypatia.catalog import CatalogQuery
from hypatia.interfaces import ICatalog
from hypatia.query import Any, Eq
from persistent import Persistent
from pyramid.events import subscriber
from sqlalchemy import and_, func
from zope.container.contained import Contained
from zope.lifecycleevent.interfaces import IObjectModifiedEvent, IObjectMovedEvent
from zope.schema.fieldproperty import FieldProperty

from onf_website.component.location.interfaces import ILocationGPSInfo, ILocationInfo, \
    ILocationInseeInfo, ILocationLocationInfo, ILocationManagerTarget, ILocationTarget, \
    LOCATION_INFO_KEY
from onf_website.reference.forest.interfaces import IForestTable
from onf_website.reference.forest.model.foret import Foret, ForetGeom, \
    PARENT_SESSION as RDF_PARENT_SESSION
from onf_website.reference.insee.model import Commune, CommuneGeom, Departement, \
    PARENT_SESSION as INSEE_PARENT_SESSION, Region
from onf_website.reference.location.interfaces.region import IRegion
from pyams_alchemy.engine import get_user_session
from pyams_catalog.query import CatalogResultSet
from pyams_content.features.checker import BaseContentChecker
from pyams_content.features.checker.interfaces import ERROR_VALUE, IContentChecker
from pyams_content.features.search.interfaces import IContextUserSearchSettings
from pyams_content.shared.common import IWfSharedContentCreator, IWfSharedContentUpdater
from pyams_content.shared.common.interfaces import IWfSharedContentFinderParams
from pyams_content.shared.common.manager import SharedToolContentFinder
from pyams_content_es.content import EsSharedToolContentFinder
from pyams_content_es.interfaces import IDocumentIndexInfo
from pyams_form.interfaces.form import IFormContextPermissionChecker
from pyams_gis.point import GeoPoint
from pyams_i18n.interfaces import II18n
from pyams_skin.layer import IPyAMSUserLayer
from pyams_utils.adapter import ContextAdapter, ContextRequestAdapter, adapter_config, \
    get_annotation_adapter
from pyams_utils.factory import factory_config
from pyams_utils.inherit import BaseInheritInfo, InheritedFieldProperty
from pyams_utils.list import boolean_iter
from pyams_utils.registry import get_utility, query_utility
from pyams_utils.request import check_request
from pyams_utils.traversing import get_parent
from pyams_utils.unicode import translate_string
from pyams_utils.url import absolute_url, canonical_url
from pyams_utils.zodb import volatile_property
from pyams_workflow.interfaces import IWorkflow, IWorkflowPublicationInfo

__docformat__ = 'restructuredtext'

from onf_website import _


@factory_config(ILocationInfo)
class LocationInfo(BaseInheritInfo, Persistent, Contained):
    """Location info persistent class"""

    target_interface = ILocationTarget
    adapted_interface = ILocationInfo

    _gps_location = FieldProperty(ILocationInfo['gps_location'])
    gps_location = InheritedFieldProperty(ILocationInfo['gps_location'])
    _forests = FieldProperty(ILocationInfo['forests'])
    forests = InheritedFieldProperty(ILocationInfo['forests'])
    _all_departments_forests = FieldProperty(ILocationInfo['all_departments_forests'])
    all_departments_forests = InheritedFieldProperty(ILocationInfo['all_departments_forests'])
    _domanials_only = FieldProperty(ILocationInfo['domanials_only'])
    domanials_only = InheritedFieldProperty(ILocationInfo['domanials_only'])
    _communals_only = FieldProperty(ILocationInfo['communals_only'])
    communals_only = InheritedFieldProperty(ILocationInfo['communals_only'])
    _cities = FieldProperty(ILocationInfo['cities'])
    cities = InheritedFieldProperty(ILocationInfo['cities'])
    _address = FieldProperty(ILocationInfo['address'])
    address = InheritedFieldProperty(ILocationInfo['address'])
    _situation = FieldProperty(ILocationInfo['situation'])
    situation = InheritedFieldProperty(ILocationInfo['situation'])
    _natural_region = FieldProperty(ILocationInfo['natural_region'])
    natural_region = InheritedFieldProperty(ILocationInfo['natural_region'])
    _departments = FieldProperty(ILocationInfo['departments'])
    departments = InheritedFieldProperty(ILocationInfo['departments'])
    _countries = FieldProperty(ILocationInfo['countries'])
    countries = InheritedFieldProperty(ILocationInfo['countries'])
    _structures = FieldProperty(ILocationInfo['structures'])
    structures = InheritedFieldProperty(ILocationInfo['structures'])
    _public_label = FieldProperty(ILocationInfo['public_label'])
    public_label = InheritedFieldProperty(ILocationInfo['public_label'])

    locations = FieldProperty(ILocationInfo['locations'])

    @volatile_property
    def get_forests_from_departments(self):
        return self.all_departments_forests or self.domanials_only or self.communals_only

    @volatile_property
    def forests_index(self):
        forests = set(self.forests or ())
        if self.departments and self.get_forests_from_departments:
            session = get_user_session(RDF_PARENT_SESSION)
            params = [Foret.departement.in_(self.selected_departments_codes)]
            if self.domanials_only:
                params.append(Foret.est_domaniale == True)
            elif self.communals_only:
                params.append(Foret.est_domaniale == False)
            results = set(*zip(*session.query(Foret.id_nat_frt)
                                       .filter(and_(*params))))
            if results:
                forests |= results
        return list(forests)

    def get_forests_list(self, params=None):
        forest_ids = (params or {}).get('id_nat_frt')
        if forest_ids:
            forest_ids = forest_ids.split(',')
        session = get_user_session(RDF_PARENT_SESSION)
        params = [Foret.id_nat_frt.in_(self.forests_index)]
        if forest_ids:
            params.append(Foret.id_nat_frt.in_(forest_ids))
        return [
            {
                'id_nat_frt': forest.id_nat_frt,
                'libelle_usage': forest.libelle_usage,
                'libelle_strict': forest.libelle_strict
            }
            for forest in session.query(Foret).filter(and_(*params))
        ]

    def get_departments(self, not_if_forests=True):
        if not_if_forests and self.get_forests_from_departments:
            return
        for value in self.departments or ():
            if IRegion.providedBy(value):
                for dept in value.departments or ():
                    yield dept
            else:
                yield value

    @volatile_property
    def selected_departments_codes(self):
        return [dept.insee_code for dept in self.get_departments(False)]

    @volatile_property
    def departments_index(self):
        return [dept.insee_code for dept in self.get_departments()]

    def get_region_and_departments(self):
        departments = set()
        forests = set(self.forests or ())
        if forests:
            session = get_user_session(RDF_PARENT_SESSION)
            departments |= set(*zip(*session.query(Foret.departement)
                                            .filter(Foret.id_nat_frt.in_(forests))))
        cities = set(self.cities or ())
        if cities:
            session = get_user_session(INSEE_PARENT_SESSION)
            departments |= set(*zip(*session.query(Commune.dep)
                                            .filter(Commune.code.in_(cities))))
        departments |= set(self.departments_index)
        if departments:
            session = get_user_session(INSEE_PARENT_SESSION)
            depts, regions = zip(*session.query(Departement.nccenr, Region.nccenr)
                                         .join(Departement, Departement.region==Region.region)
                                         .filter(Departement.dep.in_(departments)))
            return {
                'dep': ','.join(set(depts)),
                'reg': ','.join(set(regions))
            }
        return None

    @volatile_property
    def countries_index(self):
        return [value.insee_code for value in self.countries or ()]

    @property
    def automatic_label(self):
        result = []
        request = check_request()
        translate = request.localizer.translate
        if self.forests:
            session = get_user_session(RDF_PARENT_SESSION)
            result.append(', '.join(*zip(*session.query(Foret.libelle_usage)
                                                 .filter(Foret.id_nat_frt.in_(self.forests)))))
        if self.departments and self.get_forests_from_departments:
            if self.all_departments_forests:
                result.append(translate(_("Forests {0}")).format(
                    ', '.join((II18n(dept).query_attribute('appropriation_title', request=request) or
                               II18n(dept).query_attribute('title', request=request)
                               for dept in self.departments))))
            elif self.domanials_only:
                result.append(translate(_("Domanial forests {0}")).format(
                    ', '.join((II18n(dept).query_attribute('appropriation_title', request=request) or
                               II18n(dept).query_attribute('title', request=request)
                               for dept in self.departments))))
            elif self.communals_only:
                result.append(translate(_("Communal forests {0}")).format(
                    ', '.join((II18n(dept).query_attribute('appropriation_title', request=request) or
                               II18n(dept).query_attribute('title', request=request)
                               for dept in self.departments))))
        if self.cities:
            session = get_user_session(INSEE_PARENT_SESSION)
            result.append(', '.join((
                commune.title_enr
                for commune in session.query(Commune).filter(Commune.code.in_(self.cities)))
            ))
        if self.departments and not self.get_forests_from_departments:
            result.append(', '.join((II18n(dept).query_attribute('title', request=request)
                                     for dept in self.departments)))
        return ' - '.join(result)

    def get_elements(self):
        """Get iterator on current location elements"""

        from onf_website.shared.forest.interfaces import FOREST_CONTENT_TYPE, IForestManager

        request = check_request()
        label = II18n(self).query_attribute('public_label', request=request)
        if label:
            yield {
                'title': label,
                'href': None
            }
            return
        catalog = get_utility(ICatalog)
        if self.forests:
            table = query_utility(IForestTable)
            manager = query_utility(IForestManager)
            if manager is not None:
                workflow = IWorkflow(manager)
                session = get_user_session(RDF_PARENT_SESSION)
                for forest_id, (forest_ref, prop_ref, infos) in \
                        Foret.find_with_infos(self.forests, session).items():
                    params = Eq(catalog['content_type'], FOREST_CONTENT_TYPE) & \
                        Eq(catalog['forests'], forest_id) & \
                        Any(catalog['workflow_state'], workflow.published_states)
                    has_results, results = boolean_iter(
                        CatalogResultSet(CatalogQuery(catalog).query(params)))
                    if has_results:
                        forest = next(results)
                        if not IWorkflowPublicationInfo(forest).is_visible(request):
                            continue
                        yield {
                            'title': '{} ({})'.format(
                                II18n(forest).query_attribute('title', request=request),
                                forest_ref.departement),
                            'href': canonical_url(forest, request)
                        }
                    else:
                        if table.forest_displays.get(prop_ref.categorie) and \
                                ((infos is None) or infos.visible):
                            href = '{}/+f/{}::{}.html'.format(
                                absolute_url(request.root, request),
                                forest_ref.id_nat_frt,
                                translate_string((infos.libelle if infos is not None else None) or
                                                 forest_ref.libelle_usage, spaces='-'))
                        else:
                            href = None
                        yield {
                            'title': '{} ({})'.format(
                                (infos.libelle if infos is not None else None) or
                                    forest_ref.libelle_usage,
                                forest_ref.departement),
                            'href': href
                        }
        if self.cities:
            session = get_user_session(INSEE_PARENT_SESSION)
            for commune in session.query(Commune).filter(Commune.code.in_(self.cities)):
                yield {
                    'title': '{} ({})'.format(commune.title_enr, commune.dep),
                    'href': None
                }
        if self.departments:
            for dept in self.departments:
                yield {
                    'title': '{} ({})'.format(II18n(dept).query_attribute('title',
                                                                          request=request),
                                              dept.insee_code),
                    'href': None
                }

    def get_gps_locations(self):
        result = []
        if self.gps_location:
            result = (self.gps_location,)
        elif self.forests:
            session = get_user_session(RDF_PARENT_SESSION)
            forests = session.query(ForetGeom) \
                .filter(ForetGeom.id_nat_frt.in_(self.forests))
            for forest in forests:
                centroid = session.scalar(func.ST_PointOnSurface(forest.geom))
                longitude = session.scalar(func.ST_X(centroid))
                latitude = session.scalar(func.ST_Y(centroid))
                result.append(GeoPoint(longitude=longitude, latitude=latitude))
        elif self.cities:
            session = get_user_session(INSEE_PARENT_SESSION)
            cities = session.query(CommuneGeom) \
                .filter(CommuneGeom.code.in_(self.cities))
            for city in cities:
                centroid = session.scalar(func.ST_Centroid(city.geom))
                longitude = session.scalar(func.ST_X(centroid))
                latitude = session.scalar(func.ST_Y(centroid))
                result.append(GeoPoint(longitude=longitude, latitude=latitude))
        return tuple(result)

    def update_gps_locations(self):
        self.locations = self.get_gps_locations()


@subscriber(IObjectMovedEvent, context_selector=ILocationTarget)
def handle_moved_location(event):
    """Remove volatile 'parent' property when location target is moved"""
    location = ILocationInfo(event.object)
    del location.parent


@subscriber(IObjectModifiedEvent, context_selector=ILocationInfo)
def handle_modified_location(event):
    """Handle modified location"""
    location = event.object
    for attributes in event.descriptions:
        if attributes.interface is ILocationLocationInfo:
            del location.get_forests_from_departments
            del location.forests_index
        elif attributes.interface is ILocationInseeInfo:
            del location.selected_departments_codes
            del location.departments_index
            del location.countries_index
    location.update_gps_locations()


@adapter_config(required=ILocationTarget,
                provides=ILocationInfo)
def location_info_factory(target):
    """Location info factory"""
    return get_annotation_adapter(target, LOCATION_INFO_KEY, ILocationInfo,
                                  name='++location++')


LOCATION_MARKER = object()


@adapter_config(name='location',
                required=ILocationTarget,
                provides=IWfSharedContentCreator)
@adapter_config(name='location',
                required=ILocationTarget,
                provides=IWfSharedContentUpdater)
class LocationTargetContentCreator(ContextAdapter):
    """Location target content creator/updater"""

    weight = 100

    def update_content(self, params, request=None):
        """Update location from params"""
        location = params.get('location')
        if not location:
            return
        info = ILocationInfo(self.context)
        gps_location = location.get('gps_location', LOCATION_MARKER)
        if gps_location is not LOCATION_MARKER:
            if gps_location:
                info.gps_location = GeoPoint(longitude=gps_location['longitude'],
                                             latitude=gps_location['latitude'])
            else:
                info.gps_location = None
        forests = location.get('forests', LOCATION_MARKER)
        if forests is not LOCATION_MARKER:
            info.forests = forests or None
            del info.forests_index
        cities = location.get('cities', LOCATION_MARKER)
        if cities is not LOCATION_MARKER:
            info.cities = cities or None
        departments = location.get('departments', LOCATION_MARKER)
        if departments is not LOCATION_MARKER:
            info.departments = departments or None
            del info.departments_index
        if 'all_departments_forests' in location:
            info.all_departments_forests = location['all_departments_forests']
        if 'domanials_only' in location:
            info.domanials_only = location['domanials_only']
        if 'communals_only' in location:
            info.communals_only = location['communals_only']


@adapter_config(required=ILocationInfo,
                provides=IFormContextPermissionChecker)
class LocationInfoPermissionChecker(ContextAdapter):
    """Location info checker"""

    @property
    def edit_permission(self):
        content = get_parent(self.context, ILocationTarget)
        return IFormContextPermissionChecker(content).edit_permission


@adapter_config(name='location',
                required=ILocationTarget,
                provides=IDocumentIndexInfo)
def location_target_index_info(content):
    """Location index info"""
    location = ILocationInfo(content)
    values = {
        'forests': location.forests_index,
        'cities': location.cities or [],
        'departments': location.departments_index,
        'countries': location.countries_index,
        'structures': location.structures or []
    }
    if location.locations:
        values['coords'] = [loc.wgs_coordinates for loc in location.locations]
    return {'location': values}


@adapter_config(name='location',
                required=ILocationTarget,
                provides=IContentChecker)
class LocationContentChecker(BaseContentChecker):
    """Location content checker"""

    label = _("Location")
    weight = 90

    def inner_check(self, request):
        output = []
        translate = request.localizer.translate
        location = ILocationInfo(self.context)
        if not (location.countries or location.departments or location.structures or
                location.address or location.cities or location.forests or location.gps_location):
            output.append(translate(ERROR_VALUE).format(
                field=translate(_("Location fields")),
                message=translate(_("no defined location"))))
        return output


@adapter_config(name='forests',
                required=SharedToolContentFinder,
                provides=IWfSharedContentFinderParams)
class SharedToolContentFinderLocationForestParams(ContextAdapter):
    """Shared tool content finder forest location params getter"""

    def get_params(self, query, params, request=None, **kwargs):
        """Search results forest location params getter"""
        catalog = kwargs.get('catalog')
        if catalog is None:
            catalog = get_utility(ICatalog)
        if 'id_nat_frt' in params:
            forest_ids = list(map(str.strip, params['id_nat_frt'].split(',')))
            query &= Any(catalog['forests'], forest_ids)
        return query


@adapter_config(name='forests',
                required=EsSharedToolContentFinder,
                provides=IWfSharedContentFinderParams)
class EsSharedToolContentFinderLocationForestParams(ContextAdapter):
    """Elasticsearch shared tool content finder forest location params getter"""

    def get_params(self, query, params, request=None):
        """Search results forest location params getter"""
        if 'id_nat_frt' in params:
            forest_ids = list(map(str.strip, params['id_nat_frt'].split(',')))
            query &= Q('terms', **{'location.forests': forest_ids})
        return query


@adapter_config(name='forest',
                required=(ILocationTarget, IPyAMSUserLayer),
                provides=IContextUserSearchSettings)
class LocationContextUserSearchSettings(ContextRequestAdapter):
    """Location context user search settings"""

    def get_settings(self):
        location = ILocationInfo(self.context)
        for forest in location.forests:
            yield 'forest', forest
