#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from persistent import Persistent
from zope.location import Location
from zope.schema.fieldproperty import FieldProperty

from onf_website.component.illustration.interfaces import IIllustrationParagraphRendererSettings
from onf_website.skin.public import jquery_tilt
from pyams_content.component.illustration.interfaces import IIllustration, IIllustrationParagraph
from pyams_content.component.paragraph.interfaces.html import IHTMLParagraph
from pyams_content.features.renderer.interfaces import IContentRenderer
from pyams_default_theme.component.illustration import BaseIllustrationRenderer
from pyams_default_theme.component.illustration.interfaces import ILLUSTRATION_AFTER_BODY, ILLUSTRATION_BEFORE_BODY, \
    ILLUSTRATION_BEFORE_TITLE
from pyams_skin.layer import IPyAMSLayer
from pyams_template.template import template_config
from pyams_utils.adapter import NullAdapter, adapter_config, get_annotation_adapter
from pyams_utils.factory import factory_config

from onf_website import _


#
# Illustration paragraph renderers settings
#

ILLUSTRATION_RENDERER_SETTINGS_KEY = 'onf_website.illustration.renderer:default'


@factory_config(IIllustrationParagraphRendererSettings)
class IllustrationParagraphRendererSettings(Persistent, Location):
    """Illustration paragraph renderer settings"""

    display_description = FieldProperty(IIllustrationParagraphRendererSettings['display_description'])
    description_position = FieldProperty(IIllustrationParagraphRendererSettings['description_position'])
    display_button = FieldProperty(IIllustrationParagraphRendererSettings['display_button'])
    button_label = FieldProperty(IIllustrationParagraphRendererSettings['button_label'])

    def __init__(self):
        super(IllustrationParagraphRendererSettings, self).__init__()
        self.button_label = {
            'en': "Read description",
            'fr': "Lire la description"
        }


@adapter_config(context=IIllustration,
                provides=IIllustrationParagraphRendererSettings)
def default_illustration_paragraph_renderer_settings_factory(context):
    """Illustration paragraph renderer settings factory"""
    return get_annotation_adapter(context, ILLUSTRATION_RENDERER_SETTINGS_KEY,
                                  IIllustrationParagraphRendererSettings)


#
# Illustration paragraph renderers
#

@adapter_config(name='banner',
                context=(IIllustration, IPyAMSLayer),
                provides=IContentRenderer)
@template_config(template='templates/illustration-banner.pt', layer=IPyAMSLayer)
class BannerIllustrationRenderer(BaseIllustrationRenderer):
    """Banner illustration renderer"""

    label = _("ONF: full width illustration banner")
    weight = 1

    position = ILLUSTRATION_BEFORE_TITLE

    def __new__(cls, context, request):
        parent = context.__parent__
        if not IHTMLParagraph.providedBy(parent):
            return None
        return BaseIllustrationRenderer.__new__(cls)


@adapter_config(name='banner',
                context=(IIllustrationParagraph, IPyAMSLayer),
                provides=IContentRenderer)
@template_config(template='templates/illustration-paragraph-banner.pt', layer=IPyAMSLayer)
class BannerIllustrationParagraphRenderer(BaseIllustrationRenderer):
    """Banner illustration paragraph renderer"""

    label = _("ONF: full width illustration banner")
    weight = 10

    position = ILLUSTRATION_BEFORE_TITLE


@adapter_config(name='centered-before-body',
                context=(IIllustration, IPyAMSLayer),
                provides=IContentRenderer)
@template_config(template='templates/illustration-default.pt', layer=IPyAMSLayer)
@template_config(name='PARA:title-left',
                 template='templates/illustration-title-left.pt', layer=IPyAMSLayer)
class CenteredBeforeTextIllustrationRenderer(BaseIllustrationRenderer):
    """Default illustration renderer"""

    label = _("ONF: centered illustration before text")
    weight = 20

    position = ILLUSTRATION_BEFORE_BODY
    settings_interface = IIllustrationParagraphRendererSettings


@adapter_config(name='left+zoom',
                context=(IIllustration, IPyAMSLayer),
                provides=IContentRenderer)
@template_config(template='templates/illustration-left.pt', layer=IPyAMSLayer)
@template_config(name='PARA:title-left',
                 template='templates/illustration-title-left.pt', layer=IPyAMSLayer)
class LeftIllustrationWithZoomRenderer(BaseIllustrationRenderer):
    """Illustration renderer with small image and zoom"""

    label = _("ONF: small illustration on the left")
    weight = 30

    position = ILLUSTRATION_BEFORE_BODY


@adapter_config(name='right+zoom',
                context=(IIllustration, IPyAMSLayer),
                provides=IContentRenderer)
@template_config(template='templates/illustration-right.pt', layer=IPyAMSLayer)
@template_config(name='PARA:title-left',
                 template='templates/illustration-title-left.pt', layer=IPyAMSLayer)
class RightIllustrationWithZoomRenderer(BaseIllustrationRenderer):
    """Illustration renderer with small image and zoom"""

    label = _("ONF: small illustration on the right")
    weight = 40

    position = ILLUSTRATION_BEFORE_BODY


@adapter_config(name='default',
                context=(IIllustration, IPyAMSLayer),
                provides=IContentRenderer)
@template_config(template='templates/illustration-default.pt', layer=IPyAMSLayer)
@template_config(name='PARA:title-left',
                 template='templates/illustration-title-left.pt', layer=IPyAMSLayer)
class CenteredIllustrationRenderer(BaseIllustrationRenderer):
    """Default illustration renderer"""

    label = _("ONF: centered illustration after text")
    weight = 50

    position = ILLUSTRATION_AFTER_BODY
    settings_interface = IIllustrationParagraphRendererSettings


@adapter_config(name='circle',
                context=(IIllustration, IPyAMSLayer),
                provides=IContentRenderer)
@template_config(template='templates/illustration-circle.pt', layer=IPyAMSLayer)
class CircleIllustrationRenderer(BaseIllustrationRenderer):
    """Default illustration renderer"""

    label = _("ONF: circular illustration before text")
    weight = 60

    position = ILLUSTRATION_BEFORE_BODY

    def __new__(cls, context, request):
        if not IHTMLParagraph.providedBy(context.__parent__):
            return None
        return BaseIllustrationRenderer.__new__(cls)


@adapter_config(name='circle',
                context=(IIllustrationParagraph, IPyAMSLayer),
                provides=IContentRenderer)
@template_config(template='templates/illustration-circle.pt', layer=IPyAMSLayer)
@template_config(name='PARA:title-left',
                 template='templates/illustration-title-left.pt', layer=IPyAMSLayer)
class CircleIllustrationParagraphRenderer(BaseIllustrationRenderer):
    """Default illustration renderer"""

    label = _("ONF: circular illustration")
    weight = 60

    position = ILLUSTRATION_BEFORE_BODY


@adapter_config(name='left+circle',
                context=(IIllustration, IPyAMSLayer),
                provides=IContentRenderer)
@template_config(template='templates/empty.pt', layer=IPyAMSLayer)
class LeftCircleIllustrationRenderer(BaseIllustrationRenderer):
    """Illustration circle in circle, left of text"""

    label = _("ONF: circular illustration on the left")
    weight = 70

    position = ILLUSTRATION_BEFORE_BODY
    resources = (jquery_tilt, )

    def __new__(cls, context, request):
        if not IHTMLParagraph.providedBy(context.__parent__):
            return None
        return BaseIllustrationRenderer.__new__(cls)


@adapter_config(name='left+circle',
                context=(IIllustrationParagraph, IPyAMSLayer),
                provides=IContentRenderer)
class LeftCircleIllustrationParagraphRenderer(NullAdapter):
    """Disabled left circle illustration paragraph renderer"""


@adapter_config(name='right+circle',
                context=(IIllustration, IPyAMSLayer),
                provides=IContentRenderer)
@template_config(template='templates/empty.pt', layer=IPyAMSLayer)
class RightCircleIllustrationRenderer(BaseIllustrationRenderer):
    """Illustration circle in circle, right of text

    WARNING: no defined template as rendering is overridden by html paragraph
    """

    label = _("ONF: circular illustration on the right")
    weight = 80

    position = ILLUSTRATION_BEFORE_BODY
    resources = (jquery_tilt, )

    def __new__(cls, context, request):
        if not IHTMLParagraph.providedBy(context.__parent__):
            return None
        return BaseIllustrationRenderer.__new__(cls)


@adapter_config(name='right+circle',
                context=(IIllustrationParagraph, IPyAMSLayer),
                provides=IContentRenderer)
class RightCircleIllustrationParagraphRenderer(NullAdapter):
    """Disabled left circle illustration paragraph renderer"""
