#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from zope.annotation.interfaces import IAttributeAnnotatable
from zope.interface import Attribute, Interface, Invalid, invariant
from zope.schema import Bool, Choice, Object, Set

from onf_website.reference.forest.schema import ForestsListField
from onf_website.reference.insee.schema import CommunesListField
from onf_website.reference.orga.schema import StructuresListField
from pyams_content.shared.view.interfaces import IViewSettings
from pyams_sequence.schema import InternalReferenceField
from pyams_utils.interfaces.inherit import IInheritInfo
from pyams_utils.schema import PersistentDict


__docformat__ = 'restructuredtext'

from onf_website import _


#
# Hearing manager settings
#

HEARING_SETTINGS = {
    'readership': _("Readership"),
    'national': _("National scope"),
    'location': _("Additional geographic scope"),
    'organization': _("ONF organization"),
    'sites': _("Diffusion sites")
}
HEARING_SETTINGS_KEYS = ('readership', 'national', 'location', 'organization', 'sites')

HEARING_SETTINGS_KEY = 'onf_website.hearing.settings'


class IHearingSetting(Interface):
    """Hearing setting state interface"""

    name = Attribute("Setting name")

    visible = Bool(title=_("Visible?"),
                   required=True,
                   default=True)

    unfolded = Bool(title=_("Unfolded?"),
                    required=True,
                    default=True)


class IHearingManagerSettings(Interface):
    """Hearing manager settings

    This interface is used to define which hearing components are available,
    and if they are enabled and displayed by default for a given content.
    """

    settings = PersistentDict(title=_("Hearing settings"),
                              key_type=Choice(values=HEARING_SETTINGS_KEYS),
                              value_type=Object(schema=IHearingSetting))


class IHearingManagerTarget(IAttributeAnnotatable):
    """Hearing manager target marker interface"""


#
# Content hearing interfaces
#

HEARING_INFO_KEY = 'onf_website.hearing'


class IHearingReadershipInfo(Interface):
    """Hearing readership interface"""

    targets = Set(title=_("Targets"),
                  description=_("Who is concerned by this content?"),
                  value_type=Choice(vocabulary='ONF targets'),
                  required=False)

    targets_index = Attribute("Targets list used for indexing")

    private = Bool(title=_("Restricted access?"),
                   description=_("If 'yes', access will be restricted to selected groups"),
                   required=False,
                   default=False)

    private_groups = Set(title=_("Authorized groups"),
                         description=_("List of groups having access to this content"),
                         value_type=Choice(vocabulary='PyAMS local groups'),
                         required=False)


class IHearingNationalInfo(Interface):
    """Hearing national scope interface"""

    national_scope = Bool(title=_("Is this content representative of ONF actions?"),
                          required=False,
                          default=False)


class IHearingLocationsInfo(Interface):
    """Hearing location scope interface"""

    forests = ForestsListField(title=_("Forests"),
                               required=False)

    departments_forests = Bool(title=_("All forests of selected departments"),
                               required=False,
                               default=False)

    domanials_only = Bool(title=_("All domanial forests"),
                          required=False,
                          default=False)

    communals_only = Bool(title=_("All communal forests"),
                          required=False,
                          default=False)

    get_forests_from_departments = Attribute("Boolean flag used to specified if forests list "
                                             "is extracted from selected departments")

    forests_index = Attribute("Forests list used for indexing")

    @invariant
    def check_forests(self):
        if len(tuple(filter(bool, (self.departments_forests,
                                   self.domanials_only,
                                   self.communals_only)))) > 1:
            raise Invalid(_("You can choose all forests, domanial forests only or "
                            "communal forests only, but not several kinds of them"))

    cities = CommunesListField(title=_("Cities"),
                               required=False)

    departments = Set(title=_("Departments"),
                      description=_("WARNING: if forests selection is based on departments "
                                    "selection, departments selected here will not be usable to "
                                    "locate this content, but only to find matching location "
                                    "forests"),
                      value_type=Choice(vocabulary='ONF regions with departments'),
                      required=False)

    def get_departments(self):
        """Get complete departments list"""

    departments_index = Attribute("Departments list used for indexing")

    countries = Set(title=_("Foreign countries"),
                    value_type=Choice(vocabulary='ONF countries'),
                    required=False)

    countries_index = Attribute("Countries list used for indexing")


class IHearingOrganizationInfo(Interface):
    """hearing organization scope interface"""

    structures = StructuresListField(title=_("Structures"),
                                     required=False)

    departments_structures = Bool(title=_("Extend to all structures of selected departments"),
                                  required=False,
                                  default=False)

    metropolitan_structures = Bool(title=_("Metropolitan structures"),
                                   required=False,
                                   default=False)

    corsican_structures = Bool(title=_("Corsican structures"),
                               required=False,
                               default=False)

    dom_structures = Bool(title=_("Overseas structures"),
                          required=False,
                          default=False)

    structures_index = Attribute("Structures list used for indexing")


class IHearingDiffusionSitesInfo(Interface):
    """Hearing diffusion sites interface"""

    source = Choice(title=_("Original site"),
                    description=_("Main source site of current content"),
                    required=False,
                    vocabulary='PyAMS shared sites')

    source_index = Attribute("Source value used for indexing")

    def get_source_site(self):
        """Return source site as object"""

    source_folder = InternalReferenceField(title=_("Source folder"),
                                           description=_("Main folder to which content is "
                                                         "attached; if this reference is set, "
                                                         "canonical URL will be based on this "
                                                         "object's URL"),
                                           required=False)

    def get_source_folder(self):
        """Return source folder as object"""

    diffusion_sites = Set(title=_("Other diffusion sites"),
                          value_type=Choice(vocabulary='PyAMS shared sites'),
                          required=False)

    diffusion_sites_index = Attribute("Diffusion sites list used for indexing")

    def get_diffusion_sites(self):
        """Return diffusion sites as iterator"""


class IHearingInfo(IInheritInfo, IHearingReadershipInfo, IHearingNationalInfo,
                   IHearingLocationsInfo, IHearingOrganizationInfo,
                   IHearingDiffusionSitesInfo, IAttributeAnnotatable):
    """Hearing information interface"""

    def is_matching_location(self, other):
        """Check if hearing info is matching location of another content"""


class IHearingTarget(IAttributeAnnotatable):
    """Hearing target interface"""


VIEW_HEARING_SETTINGS_KEY = 'onf_website.view.hearing'


class IViewHearingSettings(IViewSettings):
    """View hearing settings"""

    select_context_targets = Bool(title=_("Select context targets?"),
                                  description=_("If 'yes', targets will be extracted from "
                                                "context hearing"),
                                  required=True,
                                  default=False)

    targets = Set(title=_("Other targets"),
                  value_type=Choice(vocabulary='ONF targets'),
                  required=False)

    def get_targets(self, context):
        """Get all targets from given context"""

    def get_targets_index(self, context):
        """Get all targets index values from given context"""

    use_context_national_scope = Bool(title=_("Use context national scope?"),
                                      description=_("If 'yes', context national scope will be "
                                                    "used as query filter"),
                                      required=True,
                                      default=False)

    national_scope = Bool(title=_("Force national scope"),
                          description=_("If 'yes', ONLY contents with national scope "
                                        "will be selected"),
                          required=True,
                          default=False)

    def get_national_scope(self, context):
        """Get view national scope filter"""

    select_context_forests = Bool(title=_("Select context forests?"),
                                  description=_("If 'yes', forests will be extracted from "
                                                "context hearing"),
                                  required=True,
                                  default=False)

    forests = ForestsListField(title=_("Other forests"),
                               required=False)

    def get_forests(self, context):
        """Get all forests from given context"""

    select_context_cities = Bool(title=_("Select context cities"),
                                 description=_("If 'yes', cities will be extracted from "
                                               "context hearing"),
                                 required=True,
                                 default=False)

    cities = CommunesListField(title=_("Other cities"),
                               required=False)

    def get_cities(self, context):
        """Get all cities from given context"""

    select_context_departments = Bool(title=_("Select context departments"),
                                      description=_("If 'yes', departments will be extracted "
                                                    "from context hearing"),
                                      required=True,
                                      default=False)

    departments = Set(title=_("Other departments"),
                      value_type=Choice(vocabulary='ONF regions with departments'),
                      required=False)

    def get_departments(self, context):
        """Get all departments from given context"""

    def get_departments_index(self, console):
        """Get all departments index values from given context"""

    select_context_countries = Bool(title=_("Select context countries"),
                                    description=_("If 'yes', countries will be extracted from "
                                                  "context hearing"),
                                    required=True,
                                    default=False)

    countries = Set(title=_("Other foreign countries"),
                    value_type=Choice(vocabulary='ONF countries'),
                    required=False)

    def get_countries(self, context):
        """Get all countries from given context"""

    def get_countries_index(self, context):
        """Get all countries index values from given context"""

    add_national_scope = Bool(title=_("Add national scope"),
                              description=_("If 'yes', contents with national scope will also "
                                            "be selected whatever is their geographic scope"),
                              required=True,
                              default=False)

    select_context_structures = Bool(title=_("Select context structures"),
                                     description=_("If 'yes', structures will be extracted "
                                                   "from context hearing"),
                                     required=True,
                                     default=False)

    structures = StructuresListField(title=_("Other structures"),
                                     required=False)

    def get_structures(self, context):
        """Get all structures from given context"""

    select_context_source = Bool(title=_("Select context source site?"),
                                 description=_("If 'yes', source site will be extracted "
                                               "from context hearing"),
                                 required=True,
                                 default=False)

    source_sites = Set(title=_("Other source sites"),
                       value_type=Choice(vocabulary='PyAMS shared sites'),
                       required=False)

    def get_source_sites(self, context):
        """Get all source sites from given context"""

    def get_source_sites_index(self, context):
        """Get all source sites index values from given context"""

    select_context_sites = Bool(title=_("Select context diffusion sites?"),
                                description=_("If 'yes', diffusion sites will be extracted "
                                              "from context hearing"),
                                required=True,
                                default=False)

    diffusion_sites = Set(title=_("Other diffusion sites"),
                          value_type=Choice(vocabulary='PyAMS shared sites'),
                          required=False)

    def get_diffusion_sites(self, context):
        """Get all diffusion sites from given context"""

    def get_diffusion_sites_index(self, context):
        """Get all diffusion sites index values from given context"""
