#
# Copyright (c) 2015-2021 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS_*** module

"""
from zope.interface import Interface
from zope.traversing.interfaces import ITraversable

from hypatia.catalog import CatalogQuery
from hypatia.interfaces import ICatalog
from hypatia.query import Eq
from pyramid.httpexceptions import HTTPFound, HTTPInternalServerError, HTTPNotFound
from pyramid.response import Response
from pyramid.view import render_view_to_response, view_config

from onf_website.reference.planning import IPlanningTable
from onf_website.reference.planning.model import PARENT_SESSION as RDF_SESSION
from onf_website.reference.planning.model.planning import PlanningData, InformationPlanning
from onf_website.shared.planning import PLANNING_CONTENT_TYPE, FakePlanning, IPlanningManager
from pyams_alchemy.engine import get_user_session
from pyams_catalog.query import CatalogResultSet
from pyams_skin.interfaces import ISkinnable
from pyams_skin.skin import apply_skin
from pyams_utils.interfaces.url import DISPLAY_CONTEXT
from pyams_utils.list import boolean_iter
from pyams_utils.registry import get_utility, query_utility
from pyams_utils.traversing import get_parent
from pyams_utils.url import absolute_url, canonical_url
from pyams_workflow.interfaces import IWorkflow, IWorkflowVersions
from pyams_utils.adapter import ContextRequestAdapter, adapter_config
from pyams_skin.layer import IPyAMSUserLayer

__docformat__ = 'restructuredtext'


@adapter_config(name='amgt',
                context=(Interface, IPyAMSUserLayer),
                provides=ITraversable)
class PlanningTraverser(ContextRequestAdapter):
    """Planning namespace traverser"""

    def traverse(self, name, furtherpath=None):
        if not name:
            raise HTTPNotFound()
        planning_id = name.upper()
        if '::' in planning_id:
            planning_id, ignored = planning_id.split('::', 1)
            if not planning_id:
                raise HTTPNotFound()
        request = self.request
        catalog = get_utility(ICatalog)
        params = Eq(catalog['content_type'], PLANNING_CONTENT_TYPE) & \
            Eq(catalog['plannings'], planning_id)
        has_results, results = boolean_iter(CatalogResultSet(CatalogQuery(catalog).query(params)))
        if has_results:
            planning = next(results)
            workflow = IWorkflow(planning, None)
            if workflow is None:
                raise HTTPInternalServerError()
            versions = IWorkflowVersions(planning, None)
            if versions is None:
                raise HTTPInternalServerError()
            planning = None
            versions = versions.get_versions(workflow.visible_states, sort=True)
            if versions:
                planning = versions[-1]
            if planning is not None:
                location = canonical_url(planning, request, query=request.params)
                if location == request.url:
                    request.annotations[DISPLAY_CONTEXT] = request.context
                    request.context = planning
                    response = render_view_to_response(planning, request, '')
                    return response
                raise HTTPFound(location=location)
        # try to fallback with fake planning content
        session = get_user_session(RDF_SESSION)
        ref_planning, info = session.query(PlanningData,
                                           InformationPlanning) \
            .outerjoin(InformationPlanning, PlanningData.id_nat_amgt == InformationPlanning.id_nat_amgt) \
            .filter(PlanningData.id_nat_amgt == planning_id) \
            .first() or (None, None)
        if ref_planning is None:
            raise HTTPNotFound()
        if (info is not None) and not info.visible:
            raise HTTPNotFound()
        table = query_utility(IPlanningTable)
        if table is None:
            raise HTTPNotFound()
        fake_planning = FakePlanning()
        fake_planning.planning_ids = [planning_id]
        return fake_planning


@view_config(route_name='planning_access')
def get_planning_access(request):
    """Direct access to given planning"""
    manager = query_utility(IPlanningManager)
    if manager is None:
        raise HTTPNotFound()
    planning_id = request.matchdict.get('id_nat_amgt')
    if not planning_id:
        raise HTTPNotFound()
    if '::' in planning_id:
        planning_id, ignored = planning_id.split('::', 1)
        if not planning_id:
            raise HTTPNotFound()
    planning_id = planning_id.upper()
    # look for content matching given planning ID
    view_name = ''.join(request.matchdict.get('view'))
    catalog = get_utility(ICatalog)
    params = Eq(catalog['content_type'], PLANNING_CONTENT_TYPE) & \
        Eq(catalog['plannings'], planning_id)
    has_results, results = boolean_iter(CatalogResultSet(CatalogQuery(catalog).query(params)))
    if has_results:
        planning = next(results)
        workflow = IWorkflow(planning, None)
        if workflow is None:
            raise HTTPInternalServerError()
        versions = IWorkflowVersions(planning, None)
        if versions is None:
            raise HTTPInternalServerError()
        planning = None
        if view_name == 'preview.html':
            planning = versions[str(versions.last_version_id)]
        else:
            versions = versions.get_versions(workflow.visible_states, sort=True)
            if versions:
                planning = versions[-1]
        if planning is not None:
            if view_name:
                location = absolute_url(planning, request, view_name)
            else:
                location = canonical_url(planning, request, query=request.params)
            if location == request.url:
                request.annotations[DISPLAY_CONTEXT] = request.context
                request.context = planning
                response = render_view_to_response(planning, request, view_name)
            else:
                response = Response()
                response.status_code = HTTPFound.code
                response.location = location
            return response
    # try to fallback with fake planning content
    session = get_user_session(RDF_SESSION)
    ref_planning, info = session.query(PlanningData, InformationPlanning) \
        .outerjoin(InformationPlanning,
                   PlanningData.id_nat_amgt == InformationPlanning.id_nat_amgt) \
        .filter(PlanningData.id_nat_amgt == planning_id) \
        .first()
    if not ref_planning:
        raise HTTPNotFound()
    if (info is not None) and not info.visible:
        raise HTTPNotFound()
    fake_planning = FakePlanning()
    fake_planning.planning_ids = [planning_id]
    target = fake_planning.__parent__
    site = get_parent(target, ISkinnable)
    apply_skin(request, ISkinnable(site).get_skin(request))
    request.annotations[DISPLAY_CONTEXT] = target
    request.context = fake_planning
    return render_view_to_response(fake_planning, request, view_name)
