#
# Copyright (c) 2015-2021 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""ONF_website.shared.planning.interfaces module

"""

from zope.interface import Interface
from zope.schema import Choice,  Dict

from onf_website.reference.forest.interfaces import FOREST_OWNERS_CATEGORIES
from onf_website.reference.planning.schema import PlanningsListField
from pyams_content.features.search import ISearchFolder
from pyams_content.shared.common import ISharedContent
from pyams_content.shared.common.interfaces import ISharedToolPortalContext, \
    IWfSharedContentPortalContext
from pyams_i18n.schema import I18nTextField
from pyams_portal.interfaces import PORTAL_TEMPLATES_VOCABULARY
from pyams_security.schema import Principal
from pyams_sequence.interfaces import IInternalReferencesList
from pyams_sequence.schema import InternalReferenceField

from pyams_i18n.schema import I18nTextLineField
from pyams_portal.interfaces import IPortletSettings

__docformat__ = 'restructuredtext'

from onf_website import _


PLANNING_CONTENT_TYPE = 'planning'
PLANNING_CONTENT_NAME = _("Planning")


MANAGE_PLANNING_PERMISSION = 'onf.ManagePlanning'
'''Permission required to manage a planning content'''


class IPlanningManagerInfo(Interface):
    """Planning manager base interface"""

    default_owner = Principal(title=_("Default owner"),
                              description=_("This principal will be set as owner of all "
                                            "plannings created automatically, except if created "
                                            "by a known contributor"))

    default_header = I18nTextField(
        title=_("Default header"),
        description=_("This header will be used as default for all plannings for which no "
                      "custom header has been defined"),
        required=False)

    default_hearing_source = Choice(title=_("Default origin site"),
                                    description=_("This site will be used as source site for all "
                                                  "plannings contents"),
                                    required=False,
                                    vocabulary='PyAMS shared sites')

    default_source_folder = InternalReferenceField(
        title=_("Default source folder"),
        description=_("Main folder to which content is attached; if this reference "
                      "is set, canonical URL will be based on this object's URL for all "
                      "plannings"),
        required=False)

    msg_expired = I18nTextField(
        title=_("Message for recently expired plannings"),
        description=_("This message is displayed for plannings that have expired less than 3 "
                      "years ago and have not been replaced"),
        required=False)

    msg_expired_3y = I18nTextField(
        title=_("Message for plannings expired more than 3 years ago"),
        description=_("This message is displayed for active but unavailable plannings"),
        required=False)

    msg_no_doc = I18nTextField(
        title=_("Message for plannings without any documents"),
        description=_("This message is displayed when an active planning doesn't contain any "
                      "document"),
        required=False)

    msg_missing_docs = I18nTextField(
        title=_("Message for plannings with missing documents"),
        description=_("This message is displayed when an active planning should have documents "
                      "which are not available in documents management system"),
        required=False)

    def get_default_source(self):
        """Get default hearing source"""

    templates = Dict(title=_("Custom templates"),
                     description=_("You can define specific templates for each planning owner "
                                   "category"),
                     key_type=Choice(vocabulary=FOREST_OWNERS_CATEGORIES),
                     value_type=Choice(vocabulary=PORTAL_TEMPLATES_VOCABULARY,
                                       required=False),
                     required=False)


class IPlanningManager(ISharedToolPortalContext, IPlanningManagerInfo):
    """Planning manager interface"""


class IPlanningManagerFactory(Interface):
    """Planning manager factory interface"""


class IWfPlanningInfo(Interface):
    """Planning information interface"""

    planning_ids = PlanningsListField(
        title=_("Plannings national IDs"),
        description=_("You can select one or several plannings which are "
                      "matching this planning content; content location "
                      "will be based automatically on this selection"),
        required=True)

    def get_planning_refs(self):
        """Get DB records matching planning IDs"""


class IWfPlanning(IWfSharedContentPortalContext, IWfPlanningInfo, IInternalReferencesList):
    """Planning interface"""


class IWfRealPlanning(Interface):
    """Real planning marker interface"""


class IWfPlanningFactory(Interface):
    """Planning interface"""


class IPlanning(ISharedContent):
    """Workflow managed planning content interface"""


class IPlanningLinkPortletSettings(IPortletSettings):

    title = I18nTextLineField(title=_("Title"),
                              description=_("If specified, this title will be displayed as title "
                                            "instead of linked planning title"),
                              required=False)

    header = I18nTextField(title=_("Header"),
                           description=_("If specified, this header will be displayed as header "
                                         "instead of linked planning header"),
                           required=False)


class IPlanningSearchFolder(ISearchFolder):
    """Planning Search folder interface"""
