
"""ONF_website.shared.planning module

"""

import locale
from collections import OrderedDict
from datetime import date

from dateutil.relativedelta import relativedelta
from pyramid.events import subscriber
from sqlalchemy.orm import joinedload
from zope.interface import implementer, provider
from zope.lifecycleevent import ObjectAddedEvent
from zope.schema.fieldproperty import FieldProperty

from onf_website.component.hearing import HearingInfo, IHearingInfo, IHearingTarget
from onf_website.component.location import ILocationInfo, ILocationTarget, LocationInfo
from onf_website.features.zfiles import IZFilesClientInfo
from onf_website.reference.forest.model.foret import Foret, ForetSurCommune
from onf_website.reference.location import IDepartmentTable
from onf_website.reference.planning.model.planning import InformationPlanning, \
    ModificationPlanning, PARENT_SESSION as RDF_SESSION, PlanningData, PlanningForet
from onf_website.shared.planning.interfaces import IPlanning, IPlanningManager, IWfPlanning, \
    IWfPlanningFactory, IWfPlanningInfo, IWfRealPlanning, PLANNING_CONTENT_NAME, \
    PLANNING_CONTENT_TYPE
from pyams_alchemy.engine import get_user_session
from pyams_content.component.illustration import IIllustrationTarget, \
    ILinkIllustrationTarget
from pyams_content.component.paragraph import IParagraphContainerTarget
from pyams_content.component.theme import ITagsInfo
from pyams_content.features.preview.interfaces import IPreviewTarget
from pyams_content.features.review import IReviewTarget
from pyams_content.shared.common import IWfSharedContentFactory, SharedContent, WfSharedContent, \
    register_content_type
from pyams_portal.interfaces import IPortalPage
from pyams_utils.adapter import ContextAdapter, adapter_config
from pyams_utils.date import format_date
from pyams_utils.interfaces import ICacheKeyValue
from pyams_utils.registry import get_utility, query_utility
from pyams_utils.request import query_request
from pyams_utils.size import get_human_size
from pyams_utils.url import absolute_url


__docformat__ = 'restructuredtext'

from onf_website import _


PLANNING_REFS_KEY = 'onf_website.planning.refs'

PLANNING_REFS_MARKER = object()

PLANNING_DOCS_NAMES = OrderedDict([
    ('id_fiche_synthese_public', _('Public synthesis document')),
    ('id_pdf_amenagement_public', _('Public planning document')),
    ('id_pdf_envoi_proprietaire', _('Owner send document')),
    ('id_pdf_deliberation_propri0', _('Owner decision document')),
    ('id_pdf_arrete_amenagement', _('Decision document'))
])

PLANNING_MODIFS_NAMES = {
    'decision': _("Update decision #{}"),
    'modification': _("Update document #{}")
}


class BasePlanningMixin:
    """Base planning mixin class"""

    content_type = PLANNING_CONTENT_TYPE
    content_name = PLANNING_CONTENT_NAME

    references = FieldProperty(IWfPlanning['references'])
    planning_ids = FieldProperty(IWfPlanning['planning_ids'])

    def get_zfiles_manager(self):
        return query_utility(IPlanningManager)

    def get_planning_refs(self):
        session = get_user_session(RDF_SESSION)
        return session.query(PlanningData, InformationPlanning) \
            .outerjoin(InformationPlanning,
                       PlanningData.id_nat_amgt == InformationPlanning.id_nat_amgt) \
            .outerjoin(ModificationPlanning,
                       PlanningData.id_nat_amgt == ModificationPlanning.id_nat_amgt) \
            .outerjoin(PlanningForet,
                       PlanningForet.id_nat_amgt == PlanningData.id_nat_amgt) \
            .outerjoin(Foret,
                       Foret.id_nat_frt == PlanningForet.id_nat_foret) \
            .filter(PlanningData.id_nat_amgt.in_(self.planning_ids)) \
            .order_by(PlanningData.categorie_proprietaire) \
            .options(joinedload(PlanningData.forets)
                .joinedload(PlanningForet.foret)
                .joinedload(Foret.communes)
                .joinedload(ForetSurCommune.commune)) \
            .options(joinedload(PlanningData.forets)
                .joinedload(PlanningForet.info_foret)) \
            .options(joinedload(PlanningData.departement_situation_insee)) \
            .options(joinedload(PlanningData.departement_situation_suppl_insee)) \
            .options(joinedload(PlanningData.modifs))

    @property
    def planning_refs(self):
        key = '{}::{}'.format(PLANNING_REFS_KEY, ICacheKeyValue(self))
        request = query_request()
        if request is not None:
            refs = request.annotations.get(key)
            if refs is not None:
                return refs
        refs = self.get_planning_refs().all()
        if request is not None:
            request.annotations[key] = refs
        return refs

    def get_document_url(self, oid, request):
        return '{}/++amgt++{}/++zfiles++{}'.format(
            absolute_url(request.root, request),
            self.planning_ids[0],
            oid)

    def get_modifs(self, request):
        translate = request.localizer.translate
        return list(
            {
                'description': translate(PLANNING_MODIFS_NAMES[modif.type_fichier])
                    .format(modif.numero_modification),
                'oid': modif.id_fichier,
                'href': self.get_document_url(modif.id_fichier, request)
            }
            for planning, info in self.planning_refs
            for modif in sorted(planning.modifs,
                                key=lambda x: (x.numero_modification,
                                               x.type_fichier))
        )

    def get_documents(self, request):
        translate = request.localizer.translate
        documents = list(
            {
                'description': translate(label),
                'oid': getattr(planning, name),
                'href': self.get_document_url(getattr(planning, name), request)
            }
            for planning, info in self.planning_refs
            for name, label in PLANNING_DOCS_NAMES.items()
            if getattr(planning, name)
        ) + self.get_modifs(request)

        if documents:
            manager = query_utility(IPlanningManager)
            client_info = IZFilesClientInfo(manager, None)
            if client_info is not None:
                zfiles_docs = dict(
                    (doc['oid'], doc)
                    for doc in client_info.get_existing_docs(doc['oid'] for doc in documents)
                )
                documents = list(filter(lambda doc: doc['oid'] in zfiles_docs, documents))
                for document in documents:
                    zfiles_doc = zfiles_docs[document['oid']]
                    document['human_size'] = get_human_size(zfiles_doc['filesize'], request)
                    document['extension'] = zfiles_doc['filename'].split('.')[-1]

        return documents

    @property
    def planning_ref(self):
        return self.planning_refs[0]

    @property
    def exp_warning(self):
        planning, info = self.planning_ref
        if planning.date_fin_applicabilite <= date.today():
            if planning.date_fin_applicabilite + relativedelta(years=3) <= date.today():
                return 'expired_3y'
            return 'expired'
        return None

    @property
    def has_docs(self):
        return len(list(filter(lambda x: x.id_pdf_amenagement_public,
                               (planning for planning, info in self.planning_refs)))) > 0

    def get_surface_cadastrale(self):
        return locale.format('%.2f',
                             sum(planning.surface_cadastrale
                                 for planning, info in self.planning_refs),
                             True)

    def get_foret_concernee(self):
        return ', '.join(sorted(set(getattr(foret.info_foret, 'libelle', None) or
                                    foret.foret.libelle_usage
                                    for planning, info in self.planning_refs
                                    for foret in planning.forets)))

    def get_departements(self):
        return ', '.join(sorted(
            set(dept.shortname
                for dept in filter(bool,
                                   (dept
                                    for depts in map(lambda x: (x.departement_situation_insee,
                                                                x.departement_situation_suppl_insee),
                                                     (planning
                                                      for planning, info in self.planning_refs))
                                    for dept in depts)))))

    def get_communes(self):
        try:
            return ', '.join(sorted(set((commune.commune.nccenr
                                         for planning, info in self.planning_refs
                                         for foret in planning.forets
                                         for commune in foret.foret.communes))))
        except AttributeError:
            return '--'

    def get_date_approbation(self):
        planning, info = self.planning_ref
        return format_date(planning.date_arrete, _('on %B %d, %Y'))

    def get_date_debut_applicabilite(self):
        planning, info = self.planning_ref
        return format_date(planning.date_debut_applicabilite, _('%B %d, %Y'))

    def get_date_fin_applicabilite(self):
        planning, info = self.planning_ref
        return format_date(planning.date_fin_applicabilite, _('%B %d, %Y'))


@implementer(IWfPlanning, IHearingTarget)
class FakePlanning(BasePlanningMixin):
    """Fake planning class"""

    __name__ = ''

    @property
    def __parent__(self):
        manager = query_utility(IPlanningManager)
        if manager is not None:
            return manager.get_default_source()

    @property
    def title(self):
        return ''.join(
            (info.libelle if info is not None else None) or planning.nom_usage_amenagement
            for planning, info in self.planning_refs
        )

    description = None

    @property
    def header(self):
        planning, info_planning = self.planning_refs[0]
        return info_planning.header if info_planning is not None else None


@adapter_config(context=FakePlanning, provides=ICacheKeyValue)
def fake_planning_cache_key(context):
    """Fake planning cache key adapter"""
    return 'AMGT::{}'.format(','.join(context.planning_ids))


@implementer(IWfPlanning, IWfRealPlanning, ILocationTarget, IHearingTarget,
             IIllustrationTarget, ILinkIllustrationTarget, IParagraphContainerTarget,
             IPreviewTarget, IReviewTarget)
class WfPlanning(BasePlanningMixin, WfSharedContent):
    """Base planning class"""


register_content_type(WfPlanning)


@provider(IWfPlanningFactory)
@implementer(IPlanning)
class Planning(SharedContent):
    """Workflow managed planning class"""


@adapter_config(context=IWfPlanningFactory, provides=IWfSharedContentFactory)
def planning_content_factory(context):
    return WfPlanning


@subscriber(ObjectAddedEvent, context_selector=IWfRealPlanning)
def real_planning_creation_handler(event):
    """Planning creation handler"""
    planning = event.object
    planning_model, planning_info = planning.get_planning_refs().first()
    if ((not planning.header) or not planning.header.get('fr')) and (planning_info is not None):
        planning.header = {'fr': planning_info.header}
    if planning_model.code_categorie_proprietaire is not None:
        templates = get_utility(IPlanningManager).templates
        owner_template = templates.get(planning_model.code_categorie_proprietaire)
        if owner_template is not None:
            page = IPortalPage(planning)
            page.inherit_parent = False
            page.shared_template = owner_template


#
# Custom planning tags adapter
#

@adapter_config(context=IWfPlanning, provides=ITagsInfo)
def planning_tags_factory(context):
    """Planning tags factory"""
    manager = query_utility(IPlanningManager)
    return ITagsInfo(manager, None)


#
# Custom planning location management
#

@adapter_config(context=IWfPlanning, provides=ILocationInfo)
class PlanningLocationInfo(ContextAdapter, LocationInfo):
    """Planning location info"""

    @property
    def _plannings(self):
        return IWfPlanningInfo(self.context).plannings

    @property
    def plannings(self):
        return self._plannings

    @property
    def _forests(self):
        return list(set((foret.id_nat_foret
                         for planning, info in self.context.planning_refs
                         for foret in planning.forets)))

    @property
    def forests(self):
        return self._forests

    @property
    def _departments(self):
        codes = set(filter(
            bool,
            (dept for depts in map(
                lambda x: (x.departement_situation,
                           x.departement_situation_suppl),
                (planning for planning, info in self.context.planning_refs))
             for dept in depts)))
        table = get_utility(IDepartmentTable)
        return [table.get(code) for code in codes]

    @property
    def departments(self):
        return self._departments


#
# Custom planning hearing management
#

@adapter_config(context=IWfPlanning, provides=IHearingInfo)
class PlanningHearingInfo(ContextAdapter, HearingInfo):
    """Planning hearing info"""

    @property
    def _plannings(self):
        return IWfPlanningInfo(self.context).plannings

    @property
    def plannings(self):
        return self._plannings

    @property
    def _source(self):
        manager = query_utility(IPlanningManager)
        if manager is not None:
            return manager.default_hearing_source

    @property
    def source(self):
        return self._source

    @property
    def source_folder(self):
        manager = query_utility(IPlanningManager)
        if manager is not None:
            return manager.default_source_folder
