#
# Copyright (c) 2008-2017 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'


# import standard library

# import interfaces
from onf_website.shared.interview.interfaces import IWfInterview, IInterviewManager, IWfInterviewInfo
from pyams_content.interfaces import CREATE_CONTENT_PERMISSION, MANAGE_CONTENT_PERMISSION
from pyams_form.interfaces.form import IWidgetForm
from pyams_i18n.interfaces import II18n
from pyams_skin.interfaces import IContentTitle, IInnerPage
from pyams_skin.interfaces.viewlet import IMenuHeader, IWidgetTitleViewletManager
from pyams_skin.layer import IPyAMSLayer
from pyams_zmi.interfaces.menu import IContentManagementMenu, IPropertiesMenu
from pyams_zmi.layer import IAdminLayer
from transaction.interfaces import ITransactionManager

# import packages
from pyams_content.shared.common.zmi import SharedContentAJAXAddForm
from pyams_content.shared.common.zmi.types import TypedSharedContentAddForm
from pyams_form.form import ajax_config
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.event import get_json_widget_refresh_event
from pyams_skin.viewlet.menu import MenuItem
from pyams_skin.viewlet.toolbar import ToolbarAction
from pyams_utils.adapter import adapter_config, ContextRequestAdapter, ContextRequestViewAdapter
from pyams_viewlet.viewlet import viewlet_config
from pyams_zmi.form import AdminEditForm
from z3c.form import field
from zope.interface import Interface, implementer

from onf_website import _


@adapter_config(context=(IWfInterview, IContentManagementMenu), provides=IMenuHeader)
class InterviewContentMenuHeader(ContextRequestAdapter):
    """Interview content menu header adapter"""

    header = _("This interview")


@adapter_config(context=(IWfInterview, IPyAMSLayer, Interface), provides=IContentTitle)
class InterviewTitleAdapter(ContextRequestViewAdapter):
    """Interview title adapter"""

    @property
    def title(self):
        translate = self.request.localizer.translate
        return translate(_("Interview « {title} »")).format(
            title=II18n(self.context).query_attribute('title', request=self.request))


@viewlet_config(name='add-shared-content.action', context=IInterviewManager, layer=IAdminLayer, view=Interface,
                manager=IWidgetTitleViewletManager, permission=CREATE_CONTENT_PERMISSION, weight=1)
class InterviewAddAction(ToolbarAction):
    """Interview adding action"""

    label = _("Add interview")
    label_css_class = 'fa fa-fw fa-plus'
    url = 'add-shared-content.html'
    modal_target = True


@pagelet_config(name='add-shared-content.html', context=IInterviewManager, layer=IPyAMSLayer,
                permission=CREATE_CONTENT_PERMISSION)
@ajax_config(name='add-shared-content.json', context=IInterviewManager, layer=IPyAMSLayer,
             permission=CREATE_CONTENT_PERMISSION, base=SharedContentAJAXAddForm)
class InterviewAddForm(TypedSharedContentAddForm):
    """Interview add form"""

    legend = _("Add interview")


#
# Custom fields properties edit form
#

@viewlet_config(name='custom-properties.menu', context=IWfInterview, layer=IPyAMSLayer,
                manager=IPropertiesMenu, permission=MANAGE_CONTENT_PERMISSION, weight=20)
class CustomPropertiesMenu(MenuItem):
    """Custom properties menu"""

    label = _("Specific properties")
    icon_class = 'fa-paperclip'
    url = '#custom-properties.html'


@pagelet_config(name='custom-properties.html', context=IWfInterview, layer=IPyAMSLayer,
                permission=MANAGE_CONTENT_PERMISSION)
@ajax_config(name='custom-properties.json', context=IWfInterview, layer=IPyAMSLayer,
             permission=MANAGE_CONTENT_PERMISSION)
@implementer(IWidgetForm, IInnerPage)
class InterviewCustomPropertiesEditForm(AdminEditForm):
    """Interview custom properties edit form"""

    legend = _("Interview specific properties")

    fields = field.Fields(IWfInterviewInfo)

    def get_ajax_output(self, changes):
        output = super(self.__class__, self).get_ajax_output(changes)
        updated = changes.get(IWfInterviewInfo, ())
        if 'portrait' in updated:
            ITransactionManager(self.context).commit()
            output.setdefault('events', []).append(
                get_json_widget_refresh_event(self.context, self.request,
                                              InterviewCustomPropertiesEditForm, 'portrait'))
        return output
