#
# Copyright (c) 2015-2022 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS_*** module

"""

import html
import json

from sqlalchemy.sql import func
from zope.interface import Interface
from zope.schema.fieldproperty import FieldProperty

from onf_website.features.search.skin import HeadFiltersSearchResultsPortletRenderer, \
    HeadFiltersSearchResultsPortletRendererSettings, SearchFolderHeaderPortletRenderer
from onf_website.features.search.skin.map import MapWithHeadFiltersSearchResultsPortletRenderer, \
    MapWithHeadFiltersSearchResultsPortletRendererSettings
from onf_website.reference.forest.model.foret import Foret, InformationForet, PARENT_SESSION as RDF_SESSION, \
    REG_ID_NAT_FRT
from onf_website.reference.insee.model import Commune, CommuneGeom, \
    PARENT_SESSION as INSEE_SESSION, REG_COMMUNE_CODE
from onf_website.shared.hub.interfaces import DISPLAY_MODE_TEMPLATE, DisplayMode, \
    IForestHubSearchFolder
from onf_website.shared.hub.skin.interfaces import \
    IForestsAndActivitiesListSearchResultsPortletRendererSettings, \
    IForestsAndActivitiesMapSearchResultsPortletRendererSettings
from onf_website.skin.public.layer import IONFBaseLayer
from pyams_alchemy.engine import get_user_session
from pyams_content.features.search.portlet import ISearchResultsPortletSettings
from pyams_content.shared.common.portlet.interfaces import ISharedContentHeaderPortletSettings
from pyams_portal.interfaces import IPortalContext, IPortalPage, IPortalTemplate, \
    IPortletRenderer, PORTAL_PAGE_KEY
from pyams_portal.page import PortalPage
from pyams_skin.layer import IPyAMSLayer
from pyams_template.template import template_config
from pyams_utils.adapter import adapter_config, get_annotation_adapter
from pyams_utils.factory import factory_config
from pyams_utils.registry import query_utility
from pyams_utils.request import query_request


__docformat__ = 'restructuredtext'

from onf_website import _


class ForestHubSearchFolderPortalPage(PortalPage):
    """Forest hub search folder portal page"""

    @property
    def template(self):
        request = query_request()
        if request is None:
            return None
        display_mode = request.params.get('display', DisplayMode.FOREST_MAP.value)
        template_attr = DISPLAY_MODE_TEMPLATE.get(display_mode)
        if template_attr and hasattr(request, 'context'):
            template_name = getattr(request.context, template_attr)
            if template_name:
                return query_utility(IPortalTemplate, name=template_name)
        return super().template


@adapter_config(context=IForestHubSearchFolder, provides=IPortalPage)
def forest_hub_search_folder_portal_page_adapter(context):
    """Forest hub search folder portal page adapter"""
    return get_annotation_adapter(context, PORTAL_PAGE_KEY, ForestHubSearchFolderPortalPage)


#
# Forests and activities map search results portlet renderer
#

@factory_config(IForestsAndActivitiesMapSearchResultsPortletRendererSettings)
class ForestsAndActivitiesMapSearchResultsPortletRendererSettings(
        MapWithHeadFiltersSearchResultsPortletRendererSettings):
    """Forest and activities search results portlet renderer settings"""

    display_header = FieldProperty(
        IForestsAndActivitiesMapSearchResultsPortletRendererSettings['display_header'])
    search_distance = FieldProperty(
        IForestsAndActivitiesMapSearchResultsPortletRendererSettings['search_distance'])
    display_distance = FieldProperty(
        IForestsAndActivitiesMapSearchResultsPortletRendererSettings['display_distance'])
    distance_message = FieldProperty(
        IForestsAndActivitiesMapSearchResultsPortletRendererSettings['distance_message'])
    short_distance_length = FieldProperty(
        IForestsAndActivitiesMapSearchResultsPortletRendererSettings['short_distance_length'])
    short_distance_message = FieldProperty(
        IForestsAndActivitiesMapSearchResultsPortletRendererSettings['short_distance_message'])
    no_result_message = FieldProperty(
        IForestsAndActivitiesMapSearchResultsPortletRendererSettings['no_result_message'])
    display_activity_switch = FieldProperty(
        IForestsAndActivitiesMapSearchResultsPortletRendererSettings['display_activity_switch'])
    display_list_switch = FieldProperty(
        IForestsAndActivitiesMapSearchResultsPortletRendererSettings['display_list_switch'])

    allow_pagination = True

    def __int__(self):
        super().__init__()
        self.distance_message = {
            'en': "At {} from selected position",
            'fr': "À une distance de {}"
        }
        self.short_distance_message = {
            'en': "Within {}",
            'fr': "À moins de {}"
        }


@adapter_config(name='forests-activities-map',
                context=(IPortalContext, IPyAMSLayer, Interface, ISearchResultsPortletSettings),
                provides=IPortletRenderer)
@template_config(template='templates/forests-activities-map.pt', layer=IPyAMSLayer)
class ForestAndActivitiesMapSearchResultsPortletRenderer(
        MapWithHeadFiltersSearchResultsPortletRenderer):
    """Forests and activities map search results portlet renderer"""

    label = _("Forests and activities search results map with head filters")

    settings_interface = IForestsAndActivitiesMapSearchResultsPortletRendererSettings
    weight = 80

    default_page_length = 20

    @property
    def values_map(self):
        user_search = self.request.params.get('user_search', '').strip()
        if user_search:
            if REG_ID_NAT_FRT.match(user_search.upper()):
                foret, info_foret = Foret.find_by_id(user_search.upper(), with_info=True).first()
                if foret is not None:
                    return json.dumps({
                        user_search: getattr(info_foret, 'libelle', None) or foret.title
                    })
            elif REG_COMMUNE_CODE.match(user_search):
                commune = Commune.get(user_search).first()
                if commune is not None:
                    return json.dumps({user_search: commune.title})
        return json.dumps({})


#
# Forests and activities list search results portlet renderer
#

@factory_config(IForestsAndActivitiesListSearchResultsPortletRendererSettings)
class ForestsAndActivitiesListSearchResultsPortletRendererSettings(
        HeadFiltersSearchResultsPortletRendererSettings):
    """Forests and activities search results portlet renderer settings"""


@adapter_config(name='forests-activities-list',
                context=(IPortalContext, IPyAMSLayer, Interface, ISearchResultsPortletSettings),
                provides=IPortletRenderer)
@template_config(template='templates/forests-activities-list.pt', layer=IPyAMSLayer)
class ForestAndActivitiesListSearchResultsPortletRenderer(
        HeadFiltersSearchResultsPortletRenderer):
    """Forests and activities list search results portlet renderer"""

    label = _("Forests and activities search results cards list with head filters")

    settings_interface = IForestsAndActivitiesListSearchResultsPortletRendererSettings
    weight = 90


#
# Forest hub search folder header portlet renderer
#

@adapter_config(context=(IForestHubSearchFolder, IONFBaseLayer, Interface,
                         ISharedContentHeaderPortletSettings),
                provides=IPortletRenderer)
class ForestHubSearchFolderHeaderPortletRenderer(SearchFolderHeaderPortletRenderer):
    """Forest hub search folder header portlet renderer"""

    @property
    def query_label(self):
        params = self.request.params
        query = params.get('user_search', '').strip()
        map_center = params.get('map_center')
        if not (query or map_center):
            return super().query_label
        translate = self.request.localizer.translate
        if map_center:
            session = get_user_session(INSEE_SESSION)
            marker = 'SRID=4326;POINT({})'.format(map_center)
            commune = session.query(Commune) \
                .join(CommuneGeom, Commune.code == CommuneGeom.code) \
                .filter(func.ST_Intersects(CommuneGeom.geom,
                                           func.ST_GeomFromText(marker))) \
                .first()
            if commune is not None:
                self.request.GET['user_search'] = commune.code
                return translate(_("Your results near “{}“")).format(html.escape(commune.title.strip()))
        if REG_ID_NAT_FRT.match(query.upper()):
            foret, info_foret = Foret.find_by_id(query.upper(), with_info=True).first()
            if foret is not None:
                self.request.GET['user_search'] = foret.id_nat_frt
                return translate(_("Your results near “{}“")).format(
                    html.escape(getattr(info_foret, 'libelle', None) or foret.title))
        elif REG_COMMUNE_CODE.match(query):
            session = get_user_session(INSEE_SESSION)
            commune = session.query(Commune) \
                .filter(Commune.code == query) \
                .first()
            if commune is not None:
                self.request.GET['user_search'] = commune.code
                return translate(_("Your results near “{}“")).format(html.escape(commune.title.strip()))
        if query:
            return translate(_("Search results for “{}“")).format(html.escape(query))
        return translate(_("Search results"))
