#
# Copyright (c) 2015-2022 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS_*** module

"""

__docformat__ = 'restructuredtext'

from datetime import datetime

from hypatia.catalog import CatalogQuery
from hypatia.interfaces import ICatalog
from hypatia.query import And, Any, Eq
from sqlalchemy.sql import and_, func, or_
from zope.schema.fieldproperty import FieldProperty

from onf_website.reference.forest.model import PARENT_SESSION as RDF_SESSION
from onf_website.reference.forest.interfaces import IForestTable
from onf_website.reference.forest.model.foret import Foret, InformationForet, ProprietaireForet
from onf_website.shared.forest import FOREST_CONTENT_TYPE, FakeForest
from onf_website.shared.hub.portlet.interfaces import IForestHubsSearchResultsPortletSettings
from pyams_alchemy.engine import get_user_session
from pyams_catalog.query import CatalogResultSet
from pyams_content.features.search.portlet import SearchResultsPortletSettings
from pyams_content.workflow import VISIBLE_STATES
from pyams_portal.portlet import Portlet, portlet_config
from pyams_utils.factory import factory_config
from pyams_utils.interfaces import VIEW_PERMISSION

from onf_website import _
from pyams_utils.registry import get_utility, query_utility
from pyams_utils.request import check_request
from pyams_utils.timezone import tztime
from pyams_utils.unicode import translate_string


FOREST_HUBS_SEARCH_PORTLET_NAME = 'onf_website.portlet.hubs.search'


@factory_config(provided=IForestHubsSearchResultsPortletSettings)
class ForestHubsSearchResultsPortletSettings(SearchResultsPortletSettings):
    """Hubs search portlet settings"""

    first_page_only = FieldProperty(IForestHubsSearchResultsPortletSettings['first_page_only'])

    def _get_items(self, request=None, start=0, length=20, limit=None, ignore_cache=False):
        start = int(start)
        length = int(length)
        if self.first_page_only and (start > 0):
            yield from iter(((), 0, {}), )
        else:
            if request is None:
                request = check_request()
            params = request.params
            user_search = params.get('user_search', '').strip().upper()
            if not user_search:
                yield from iter(((), 0, {}), )
            else:
                translated_search = translate_string(user_search, force_lower=False)
                now = tztime(datetime.utcnow())
                table = query_utility(IForestTable)
                params = [
                    Foret.date_debut_validite <= now,
                    Foret.date_fin_validite >= now,
                    ProprietaireForet.categorie.in_(table.visible_owners_ids),
                    or_(
                        Foret.libelle_recherche.contains(translated_search),
                        func.upper(InformationForet.libelle).contains(user_search),
                        func.upper(InformationForet.libelle).contains(translated_search)
                    )
                ]
                session = get_user_session(RDF_SESSION)
                query = session.query(Foret, InformationForet) \
                    .join(ProprietaireForet, Foret.id_nat_frt == ProprietaireForet.id_nat_frt) \
                    .outerjoin(InformationForet, Foret.id_nat_frt == InformationForet.id_nat_frt) \
                    .filter(and_(*params)) \
                    .distinct() \
                    .order_by(Foret.libelle_usage)

                catalog = get_utility(ICatalog)
                results, count, aggregations = [], 0, {}
                results_ids = set()
                for foret, info in query:
                    index = -1
                    if foret.id_nat_frt not in results_ids:
                        for index, item in enumerate(CatalogResultSet(CatalogQuery(catalog).query(
                                And(Eq(catalog['content_type'], FOREST_CONTENT_TYPE),
                                    Eq(catalog['forests'], foret.id_nat_frt),
                                    Any(catalog['workflow_state'], VISIBLE_STATES))))):
                            if (item not in results) and (foret.id_nat_frt not in results_ids):
                                results.append(item)
                                results_ids.add(foret.id_nat_frt)
                        else:
                            if (index < 0) and ((info is None) or info.visible):
                                results.append(FakeForest(foret.id_nat_frt))
                                results_ids.add(foret.id_nat_frt)
                count = len(results)
                yield from iter((results[start:start + length], count, aggregations), )


@portlet_config(permission=VIEW_PERMISSION)
class ForestHubsSearchResultsPortlet(Portlet):
    """Forest hubs search portlet"""

    name = FOREST_HUBS_SEARCH_PORTLET_NAME
    label = _("Forest hubs search portlet")

    toolbar_css_class = 'fa fa-fw fa-2x fa-tree'

    settings_factory = IForestHubsSearchResultsPortletSettings
