#
# Copyright (c) 2015-2021 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS_*** module

"""

from hypatia.catalog import CatalogQuery
from hypatia.interfaces import ICatalog
from hypatia.query import Any, Eq
from pyramid.events import subscriber
from z3c.form.button import Button, Buttons, handler
from z3c.form.field import Fields
from z3c.form.interfaces import DISPLAY_MODE, IDataExtractedEvent
from zope.dublincore.interfaces import IZopeDublinCore
from zope.interface import Interface, Invalid, alsoProvides, implementer

from onf_website.shared.forest import FOREST_CONTENT_TYPE, IWfForest, WfForest
from onf_website.shared.forest.interfaces import IForestManager
from onf_website.skin.zmi import onf_website
from pyams_content.component.gallery.zmi import GalleryMenu
from pyams_content.interfaces import CREATE_CONTENT_PERMISSION, MANAGE_CONTENT_PERMISSION
from pyams_content.shared.common.zmi import SharedContentAJAXAddForm, SharedContentAddForm
from pyams_content.shared.common.zmi.properties import SharedContentPropertiesEditForm
from pyams_content.zmi import pyams_content
from pyams_form.form import ajax_config
from pyams_form.interfaces.form import IWidgetForm, check_submit_button
from pyams_form.schema import ResetButton
from pyams_i18n.interfaces import II18n
from pyams_i18n.widget import I18nSEOTextLineFieldWidget
from pyams_pagelet.pagelet import pagelet_config
from pyams_security.utility import get_principal
from pyams_skin.event import get_json_form_refresh_event
from pyams_skin.interfaces import IContentTitle, IInnerPage
from pyams_skin.interfaces.tinymce import ITinyMCEConfiguration
from pyams_skin.interfaces.viewlet import IMenuHeader, IWidgetTitleViewletManager
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.viewlet.menu import MenuItem
from pyams_skin.viewlet.toolbar import ToolbarAction
from pyams_utils.adapter import ContextRequestAdapter, ContextRequestViewAdapter, NullAdapter, \
    adapter_config
from pyams_utils.date import format_datetime
from pyams_utils.fanstatic import get_resource_path
from pyams_utils.html import html_to_text
from pyams_utils.interfaces import VIEW_SYSTEM_PERMISSION
from pyams_utils.interfaces.data import IObjectData
from pyams_utils.registry import get_utility
from pyams_utils.traversing import get_parent
from pyams_viewlet.viewlet import viewlet_config
from pyams_zmi.form import AdminEditForm
from pyams_zmi.interfaces.menu import IContentManagementMenu, IPropertiesMenu
from pyams_zmi.layer import IAdminLayer


__docformat__ = 'restructuredtext'

from onf_website import _


@adapter_config(context=(IWfForest, IContentManagementMenu), provides=IMenuHeader)
class ForestContentMenuHeader(ContextRequestAdapter):
    """Forest content menu header adapter"""

    header = _("This forest")


@adapter_config(context=(IWfForest, IPyAMSLayer, Interface), provides=IContentTitle)
class ForestTitleAdapter(ContextRequestViewAdapter):
    """Forest title adapter"""

    @property
    def title(self):
        translate = self.request.localizer.translate
        return translate(_("Forest « {title} »")).format(
            title=II18n(self.context).query_attribute('title', request=self.request))


@viewlet_config(name='add-shared-content.action',
                context=IForestManager, layer=IAdminLayer, view=Interface,
                manager=IWidgetTitleViewletManager, weight=1,
                permission=CREATE_CONTENT_PERMISSION)
class ForestAddAction(ToolbarAction):
    """Forest adding action"""

    label = _("Add forest")
    label_css_class = 'fa fa-fw fa-plus'
    url = 'add-shared-content.html'
    modal_target = True


@pagelet_config(name='add-shared-content.html',
                context=IForestManager, layer=IPyAMSLayer,
                permission=CREATE_CONTENT_PERMISSION)
@ajax_config(name='add-shared-content.json',
             context=IForestManager, request_type=IPyAMSLayer,
             permission=CREATE_CONTENT_PERMISSION, base=SharedContentAJAXAddForm)
class ForestAddForm(SharedContentAddForm):
    """Forest add form"""

    legend = _("Add forest")

    fields = Fields(IWfForest).select('title', 'forest_ids', 'notepad')
    fields['title'].widgetFactory = I18nSEOTextLineFieldWidget


@subscriber(IDataExtractedEvent, form_selector=ForestAddForm)
def handle_forest_add_form_data(event):
    """Check for previous forest with same forest IDs"""
    forest_ids = event.data.get('forest_ids')
    catalog = get_utility(ICatalog)
    params = Eq(catalog['content_type'], FOREST_CONTENT_TYPE) & \
        Any(catalog['forests'], forest_ids)
    count, results = CatalogQuery(catalog).query(params)
    if count > 0:
        event.form.widgets.errors += (Invalid(_("A content already exists for this forest!")),)


@pagelet_config(name='properties.html',
                context=IWfForest, layer=IPyAMSLayer,
                permission=VIEW_SYSTEM_PERMISSION)
@ajax_config(name='properties.json',
             context=IWfForest, request_type=IPyAMSLayer,
             permission=MANAGE_CONTENT_PERMISSION)
class ForestPropertiesEditForm(SharedContentPropertiesEditForm):
    """Forest properties edit form"""

    interface = IWfForest
    fieldnames = ('title', 'short_name', 'content_url', 'forest_ids',
                  'header', 'description', 'notepad')


@viewlet_config(name='custom-properties.menu',
                context=IWfForest, layer=IAdminLayer,
                manager=IPropertiesMenu, weight=12,
                permission=VIEW_SYSTEM_PERMISSION)
class ForestCustomPropertiesMenu(MenuItem):
    """Forest custom properties menu"""

    label = _("baseline-menu", default=_("Baseline"))
    icon_class = 'fa-paperclip'
    url = '#custom-properties.html'


class IForestCustomPropertiesEditFormButtons(Interface):
    """Forest custom properties edit form buttons"""

    reset = ResetButton(name='reset', title=_("Reset"))
    save = Button(name='save', title=_("Save published version"),
                  condition=check_submit_button)
    accept = Button(name='accept', title=_("Accept proposed version"),
                    condition=check_submit_button)


@pagelet_config(name='custom-properties.html',
                context=IWfForest, layer=IPyAMSLayer,
                permission=VIEW_SYSTEM_PERMISSION)
@ajax_config(name='custom-properties.json',
             context=IWfForest, layer=IPyAMSLayer,
             permission=MANAGE_CONTENT_PERMISSION)
@implementer(IWidgetForm, IInnerPage)
class ForestCustomPropertiesEditForm(AdminEditForm):
    """Forest custom properties edit form"""

    legend = _("Forest baseline text")

    @property
    def fields(self):
        fields = Fields(IWfForest).select('baseline_draft', 'baseline_principal_status',
                                          'baseline_published')
        draft = II18n(self.context).query_attribute('baseline_draft', request=self.request)
        if not draft:
            fields = fields.omit('baseline_draft', 'baseline_principal_status')
        return fields

    buttons = Buttons(IForestCustomPropertiesEditFormButtons)
    handleActions = True

    def updateActions(self):
        super().updateActions()
        accept = self.actions.get('accept')
        if accept is not None:
            draft = II18n(self.context).query_attribute('baseline_draft', request=self.request)
            if not draft:
                accept.addClass('hidden')
            else:
                accept.addClass('btn-info')
                accept.object_data = {
                    'ams-plugins': 'onf_website',
                    'ams-plugin-onf_website-src': get_resource_path(onf_website),
                    'ams-click-handler': 'ONF_website.forest.acceptProposedBaseline'
                }
                alsoProvides(accept, IObjectData)
        save = self.actions.get('save')
        if save is not None:
            save.addClass('btn-primary')

    def updateWidgets(self, prefix=None):
        super().updateWidgets(prefix)
        draft = self.widgets.get('baseline_draft')
        if draft is not None:
            draft.set_mode(DISPLAY_MODE)
        principal = self.widgets.get('baseline_principal_status')
        if principal is not None:
            translate = self.request.localizer.translate
            baseline_draft = II18n(self.context).query_attribute('baseline_draft',
                                                                 request=self.request)
            if baseline_draft:
                principal.value = translate(_("{date} by {principal}")).format(
                    date=format_datetime(self.context.baseline_timestamp),
                    principal=get_principal(self.request, self.context.baseline_principal).title)
                if draft is not None:
                    ts = self.context.baseline_timestamp
                    if ts:
                        dc = IZopeDublinCore(self.context, None)
                        if dc.modified and (ts == dc.modified):
                            draft.after_widget_notice = \
                                '<div class="display-flex">' \
                                '<i class="fa fa-fw fa-circle txt-color-orange align-top ' \
                                'padding-top-5 padding-x-5"></i>' \
                                '<div class="display-inline text-warning">{}</div>' \
                                '</div>'.format(
                                    translate(_("Baseline was modified since last update")))
            else:
                principal.value = translate(_("No text proposal was found."))
        published = self.widgets.get('baseline_published')
        if published is not None:
            translate = self.request.localizer.translate
            published_baseline = II18n(self.context).query_attribute('baseline_published',
                                                                     request=self.request)
            current_length = len(html_to_text(published_baseline or ''))
            if current_length:
                published.after_widget_notice = \
                    translate(_('<span class="text-info">Current text '
                                'length: {} characters</span>')).format(current_length)
                manager = get_parent(self.context, IForestManager)
                max_length = manager.baseline_max_length
                if current_length > max_length:
                    published.after_widget_notice += \
                        translate(_('<strong class="text-danger padding-left-5">/ +{:n}</strong>')) \
                            .format(current_length - max_length)

    @handler(buttons['save'])
    def handleSave(self, action):
        data, errors = self.extractData()
        if errors:
            self.status = self.formErrorsMessage
            return
        changes = super().applyChanges(data)
        if changes:
            self.status = self.successMessage
        else:
            self.status = self.noChangesMessage
        return changes

    def get_ajax_output(self, changes):
        output = super().get_ajax_output(changes)
        if changes:
            output.setdefault('events', []).append(
                get_json_form_refresh_event(self.context, self.request,
                                            ForestCustomPropertiesEditForm))
        return output


@adapter_config(context=(ForestCustomPropertiesEditForm, IAdminLayer),
                provides=ITinyMCEConfiguration)
class ForestCustomPropertiesEditorConfiguration(ContextRequestAdapter):
    """Forest custom properties editor configuration"""

    configuration = {
        'ams-plugins': 'pyams_content',
        'ams-plugin-pyams_content-src': get_resource_path(pyams_content),
        'ams-plugin-pyams_content-async': 'false',
        'ams-tinymce-init-callback': 'PyAMS_content.TinyMCE.initEditor',
        'ams-tinymce-menubar': False,
        'ams-tinymce-plugins': ['paste', 'lists', 'charmap'],
        'ams-tinymce-toolbar': 'undo redo | pastetext | bold italic | bullist numlist',
        'ams-tinymce-toolbar1': False,
        'ams-tinymce-toolbar2': False,
        'ams-tinymce-height': 150
    }


#
# Medias gallery
#

@viewlet_config(name='gallery.menu',
                context=IWfForest, layer=IPyAMSLayer,
                manager=IPropertiesMenu, permission=VIEW_SYSTEM_PERMISSION, weight=15)
class ForestGalleryMenu(GalleryMenu):
    """Forest gallery menu item"""

    label = _("Presentation gallery...")


#
# Custom forest menus
#

@viewlet_config(name='location.menu', context=WfForest, layer=IAdminLayer,
                manager=IPropertiesMenu, permission=VIEW_SYSTEM_PERMISSION, weight=310)
class ForestLocationMenu(NullAdapter):
    """Hidden forest location menu"""


@viewlet_config(name='hearing.menu', context=WfForest, layer=IAdminLayer,
                manager=IPropertiesMenu, permission=VIEW_SYSTEM_PERMISSION, weight=310)
class ForestHearingMenu(NullAdapter):
    """Hidden forest hearing menu"""
