#
# Copyright (c) 2015-2021 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS_*** module

"""

from zope.interface import Interface

from onf_website.shared.forest import FakeForest, IForestManager
from pyams_portal.interfaces import IPortalPage, IPortalTemplate
from pyams_portal.page import PortalPage
from pyams_skin.interfaces.metas import IHTMLContentMetas
from pyams_skin.layer import IPyAMSUserLayer
from pyams_skin.metas import HTMLTagMeta
from pyams_utils.adapter import ContextAdapter, ContextRequestAdapter, ContextRequestViewAdapter, \
    adapter_config
from pyams_utils.interfaces.url import ICanonicalURL, IRelativeURL
from pyams_utils.registry import query_utility
from pyams_utils.unicode import translate_string
from pyams_utils.url import absolute_url

__docformat__ = 'restructuredtext'


@adapter_config(context=FakeForest, provides=IPortalPage)
class FakeForestPortalPage(ContextAdapter, PortalPage):
    """Fake forest portal page"""

    can_inherit = False

    @property
    def template(self):
        manager = query_utility(IForestManager)
        if manager is None:
            return None
        foret, prop, info = self.context.get_forest_refs().first()
        if (foret is None) or (prop is None):
            return None
        template = manager.templates.get(prop.categorie)
        if template is not None:
            return query_utility(IPortalTemplate, name=template)
        return IPortalPage(manager).template


@adapter_config(name='title',
                context=(FakeForest, IPyAMSUserLayer, Interface),
                provides=IHTMLContentMetas)
class FakeForestTitleMetasAdapter(ContextRequestViewAdapter):
    """Title metas adapter"""

    order = 1

    def get_metas(self):
        """Get HTML metas headers"""
        yield HTMLTagMeta('title', self.context.title)


@adapter_config(context=(FakeForest, IPyAMSUserLayer),
                provides=ICanonicalURL)
class FakeForestCanonicalURL(ContextRequestAdapter):
    """Fake forest canonical URL adapter"""

    def get_url(self, view_name=None, query=None):
        context = None
        manager = query_utility(IForestManager)
        if manager is not None:
            context = manager.get_default_source()
        if context is None:
            return '{}/+f/{}::{}.html'.format(absolute_url(self.request.root, self.request),
                                              ''.join(self.context.forest_ids).lower(),
                                              translate_string(self.context.title, spaces='-'))
        return '{}/++frt++{}::{}.html'.format(absolute_url(context, self.request),
                                              ''.join(self.context.forest_ids).lower(),
                                              translate_string(self.context.title, spaces='-'))


@adapter_config(context=(FakeForest, IPyAMSUserLayer),
                provides=IRelativeURL)
class FakeForestRelativeURL(ContextRequestAdapter):
    """Fake forest relative URL adapter"""

    def get_url(self, display_context=None, view_name=None, query=None):
        return '{}/++frt++{}::{}.html'.format(absolute_url(display_context, self.request),
                                              ''.join(self.context.forest_ids).lower(),
                                              translate_string(self.context.title, spaces='-'))
