#
# Copyright (c) 2015-2021 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""ONF_website.shared.forest.manager module

"""

from pyramid.events import subscriber
from pyramid.threadlocal import get_current_registry
from zope.component.interfaces import ISite
from zope.interface import implementer
from zope.lifecycleevent import ObjectAddedEvent
from zope.lifecycleevent.interfaces import IObjectAddedEvent
from zope.schema.fieldproperty import FieldProperty

from onf_website.shared.forest import Forest
from onf_website.shared.forest.interfaces import DOM_ILLUSTRATION_KEY, FOREST_CONTENT_TYPE, \
    IForestManager, IForestManagerFactory
from pyams_content.component.illustration import IIllustration, IIllustrationTarget
from pyams_content.component.paragraph import IParagraphFactorySettings
from pyams_content.component.theme import IThemesManagerTarget
from pyams_content.reference.pictograms.interfaces import IPictogramManagerTarget
from pyams_content.shared.common import ISharedSite
from pyams_content.shared.common.interfaces import ISharedContentFactory
from pyams_content.shared.common.manager import SharedTool
from pyams_sequence.reference import get_reference_target
from pyams_utils.adapter import adapter_config, get_annotation_adapter
from pyams_utils.registry import query_utility, utility_config
from pyams_utils.traversing import get_parent


__docformat__ = 'restructuredtext'


@implementer(IForestManager, IParagraphFactorySettings, IIllustrationTarget,
             IThemesManagerTarget, IPictogramManagerTarget)
class ForestManager(SharedTool):
    """Forest manager class"""

    shared_content_type = FOREST_CONTENT_TYPE

    allowed_paragraphs = FieldProperty(IParagraphFactorySettings['allowed_paragraphs'])
    auto_created_paragraphs = FieldProperty(IParagraphFactorySettings['auto_created_paragraphs'])

    baseline_help = FieldProperty(IForestManager['baseline_help'])
    baseline_max_length = FieldProperty(IForestManager['baseline_max_length'])
    gallery_help = FieldProperty(IForestManager['gallery_help'])
    default_owner = FieldProperty(IForestManager['default_owner'])
    default_hearing_source = FieldProperty(IForestManager['default_hearing_source'])
    default_source_folder = FieldProperty(IForestManager['default_source_folder'])

    def get_default_source(self):
        """Get default hearing source"""
        target = get_reference_target(self.default_source_folder)
        if target is None:
            target = query_utility(ISharedSite, name=self.default_hearing_source)
        return target

    templates = FieldProperty(IForestManager['templates'])


@utility_config(provides=IForestManagerFactory)
class ForestManagerFactory:
    """Forest manager factory"""

    def __new__(cls):
        return ForestManager


@adapter_config(context=IForestManager, provides=ISharedContentFactory)
def forest_manager_content_factory(context):
    return Forest


@subscriber(IObjectAddedEvent, context_selector=IForestManager)
def handle_added_forest_manager(event):
    """Register forest manager when added"""
    site = get_parent(event.newParent, ISite)
    registry = site.getSiteManager()
    if registry is not None:
        registry.registerUtility(event.object, IForestManager)


@adapter_config(name='dom',
                context=IForestManager, provides=IIllustration)
def forest_manager_dom_illustration_factory(context):
    """Forest manager DOM illustration factory"""

    def illustration_callback(illustration):
        get_current_registry().notify(ObjectAddedEvent(illustration, context,
                                                       illustration.__name__))

    return get_annotation_adapter(context, DOM_ILLUSTRATION_KEY, IIllustration,
                                  name='++illustration++dom',
                                  callback=illustration_callback)
