#
# Copyright (c) 2008-2019 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from pyramid.decorator import reify
from pyramid.events import subscriber
from pyramid.view import view_config
from z3c.form import field
from z3c.form.interfaces import DISPLAY_MODE, IDataExtractedEvent
from z3c.table.column import GetAttrColumn
from z3c.table.interfaces import IColumn, IValues
from zope.interface import Interface, Invalid

from onf_website.reference.insee.model import Commune
from onf_website.reference.orga.model import Structure
from onf_website.shared.contact import IContactManager
from onf_website.shared.contact.interfaces.theme import IContactTheme, IContactThemeManager, \
    IContactThemeManagerTarget
from onf_website.shared.contact.theme import ContactTheme
from pyams_content.interfaces import MANAGE_TOOL_PERMISSION
from pyams_content.zmi import pyams_content
from pyams_form.form import AJAXAddForm, ajax_config
from pyams_form.interfaces.form import IInnerSubForm
from pyams_form.security import ProtectedFormObjectMixin
from pyams_i18n.interfaces import II18n
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.container import ContainerView, delete_container_element
from pyams_skin.event import get_json_table_row_refresh_event
from pyams_skin.interfaces.container import ITableElementEditor
from pyams_skin.interfaces.viewlet import IToolbarViewletManager, IWidgetTitleViewletManager
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.table import ActionColumn, BaseTable, I18nColumn, TrashColumn
from pyams_skin.viewlet.menu import MenuItem
from pyams_skin.viewlet.toolbar import JsToolbarAction, ToolbarAction
from pyams_utils.adapter import ContextRequestViewAdapter, adapter_config
from pyams_utils.fanstatic import get_resource_path
from pyams_utils.interfaces import VIEW_SYSTEM_PERMISSION
from pyams_utils.registry import query_utility
from pyams_utils.traversing import get_parent
from pyams_utils.unicode import translate_string
from pyams_utils.url import absolute_url
from pyams_viewlet.viewlet import viewlet_config
from pyams_zmi.form import AdminDialogAddForm, AdminDialogDisplayForm, AdminDialogEditForm
from pyams_zmi.interfaces.menu import IPropertiesMenu
from pyams_zmi.layer import IAdminLayer
from pyams_zmi.view import InnerAdminView
from pyams_zmi.zmi.table import InnerTableView


__docformat__ = 'restructuredtext'

from onf_website import _


#
# Contact themes manager view
#

@viewlet_config(name='contact-themes.menu', context=IContactThemeManagerTarget, layer=IAdminLayer,
                manager=IPropertiesMenu, permission=MANAGE_TOOL_PERMISSION, weight=620)
class ContactThemeManagerMenu(MenuItem):
    """Contact themes manager menu"""

    label = _("Contact themes")
    icon_class = 'fa-tags'
    url = '#contact-themes.html'


class ContactThemeManagerTable(ProtectedFormObjectMixin, BaseTable):
    """Contact themes manager table"""

    prefix = 'themes'
    title = _("Content themes")

    sortOn = None

    @property
    def cssClasses(self):
        classes = ['table', 'table-bordered', 'table-striped', 'table-hover',
                   'table-tight', 'datatable']
        return {'table': ' '.join(classes)}

    @reify
    def data_attributes(self):
        attributes = super().data_attributes
        attributes['table'] = {
            'id': self.id,
            'data-ams-plugins': 'pyams_content',
            'data-ams-plugin-pyams_content-src': get_resource_path(pyams_content),
            'data-ams-location': absolute_url(IContactThemeManager(self.context), self.request)
        }
        return attributes

    @reify
    def values(self):
        return list(super().values)


@adapter_config(context=(IContactThemeManagerTarget, IPyAMSLayer, ContactThemeManagerTable),
                provides=IValues)
class ContactThemeManagerValues(ContextRequestViewAdapter):
    """Contact themes manager values adapter"""

    @property
    def values(self):
        return IContactThemeManager(self.context).values()


@adapter_config(name='name',
                context=(IContactThemeManagerTarget, IPyAMSLayer, ContactThemeManagerTable),
                provides=IColumn)
class ContactThemeManagerTableInternalLabelColumn(I18nColumn, GetAttrColumn):
    """Contact themes manager name column"""

    _header = _("Internal label")
    attrName = 'name'
    weight = 10

    dt_sort_type = 'string'

    def getValue(self, obj):
        value = obj.internal_label
        if not value:
            value = '<i>{}</i>'.format(
                super(ContactThemeManagerTableInternalLabelColumn, self).getValue(obj))
        return value


@adapter_config(name='label',
                context=(IContactThemeManagerTarget, IPyAMSLayer, ContactThemeManagerTable),
                provides=IColumn)
class ContactThemeManagerTableLabelColumn(I18nColumn, GetAttrColumn):
    """Contact themes manager name column"""

    _header = _("Selection label")
    attrName = 'name'
    weight = 12

    dt_sort_type = 'string'

    def getValue(self, obj):
        i18n = II18n(obj)
        value = i18n.query_attribute('label', request=self.request)
        if not value:
            value = '<i>{}</i>'.format(
                super(ContactThemeManagerTableLabelColumn, self).getValue(obj))
        return value


@adapter_config(name='contacts',
                context=(IContactThemeManagerTarget, IPyAMSLayer, ContactThemeManagerTable),
                provides=IColumn)
class ContactThemeManagerTableContactsColumn(ActionColumn):
    """Contact themes manager contacts column"""

    weight = 100

    icon_class = 'fa fa-fw fa-users'
    icon_hint = _("Theme assignments")

    url = 'contacts-dialog.html'
    modal_target = True


@adapter_config(name='associations',
                context=(IContactThemeManagerTarget, IPyAMSLayer, ContactThemeManagerTable),
                provides=IColumn)
class ContactThemeManagerTableAssociationsColumn(ActionColumn):
    """Contact themes manager associations column"""

    weight = 110

    icon_class = 'fa fa-fw fa-link'
    icon_hint = _("Theme associations")

    url = 'associations-dialog.html'
    modal_target = True

    permission = MANAGE_TOOL_PERMISSION


@adapter_config(name='trash',
                context=(IContactThemeManagerTarget, IPyAMSLayer, ContactThemeManagerTable),
                provides=IColumn)
class ContactThemeManagerTableTrashColumn(TrashColumn):
    """Contact themes manager table trash column"""

    permission = MANAGE_TOOL_PERMISSION


@view_config(name='delete-element.json', context=IContactThemeManager, request_type=IPyAMSLayer,
             permission=MANAGE_TOOL_PERMISSION, renderer='json', xhr=True)
def delete_data_type(request):
    """Contact theme delete view"""
    return delete_container_element(request, ignore_permission=True)


@pagelet_config(name='contact-themes.html', context=IContactThemeManagerTarget, layer=IPyAMSLayer,
                permission=MANAGE_TOOL_PERMISSION)
class ContactThemeManagerView(InnerAdminView, ContainerView):
    """Contact themes manager view"""

    table_class = ContactThemeManagerTable


#
# Contact themes views
#

@viewlet_config(name='add-contact-theme.action', context=IContactThemeManagerTarget,
                layer=IAdminLayer, view=ContactThemeManagerTable, manager=IWidgetTitleViewletManager,
                permission=MANAGE_TOOL_PERMISSION, weight=1)
class ContactThemeAddAction(ToolbarAction):
    """Contact theme add action"""

    label = _("Add contact theme")
    label_css_class = 'fa fa-fw fa-plus'

    url = 'add-contact-theme.html'
    modal_target = True


@pagelet_config(name='add-contact-theme.html', context=IContactThemeManagerTarget,
                layer=IPyAMSLayer, permission=MANAGE_TOOL_PERMISSION)
@ajax_config(name='add-contact-theme.json', context=IContactThemeManagerTarget,
             layer=IPyAMSLayer, base=AJAXAddForm)
class ContactThemeAddForm(AdminDialogAddForm):
    """Contact theme add form"""

    legend = _("Add new contact theme")
    icon_css_class = 'fa fa-fw fa-tags'
    dialog_class = 'modal-large'

    fields = field.Fields(IContactTheme).omit('__parent__', '__name__')
    edit_permission = MANAGE_TOOL_PERMISSION

    def create(self, data):
        return ContactTheme()

    def add(self, object):
        name = translate_string(object.name, spaces='-')
        IContactThemeManager(self.context)[name] = object

    def nextURL(self):
        return '#contact-themes.html'


@subscriber(IDataExtractedEvent, form_selector=ContactThemeAddForm)
def handle_contact_theme_add_form_data_extraction(event):
    """Check new theme for existing name"""
    context = event.form.context
    manager = IContactThemeManager(context)
    name = event.data.get('name')
    if translate_string(name, spaces='-') in manager:
        event.form.widgets.errors += (Invalid(_("Specified contact theme name is already "
                                                "used!")),)


@pagelet_config(name='properties.html', context=IContactTheme, layer=IPyAMSLayer,
                permission=MANAGE_TOOL_PERMISSION)
@ajax_config(name='properties.json', context=IContactTheme, layer=IPyAMSLayer)
class ContactThemeEditForm(AdminDialogEditForm):
    """Contact theme properties edit form"""

    prefix = 'contact_theme_properties.'

    legend = _("Contact theme properties")
    icon_css_class = 'fa fa-fw fa-tags'
    dialog_class = 'modal-large'

    fields = field.Fields(IContactTheme).omit('__parent__', '__name__')
    edit_permission = MANAGE_TOOL_PERMISSION

    def updateWidgets(self, prefix=None):
        super(ContactThemeEditForm, self).updateWidgets(prefix)
        if 'name' in self.widgets:
            self.widgets['name'].mode = DISPLAY_MODE

    def get_ajax_output(self, changes):
        output = super(ContactThemeEditForm, self).get_ajax_output(changes)
        if changes:
            context = get_parent(self.context, IContactThemeManagerTarget)
            output.setdefault('events', []).append(
                get_json_table_row_refresh_event(context,
                                                 self.request,
                                                 ContactThemeManagerView.table_class,
                                                 self.context))
        return output


#
# Theme contacts table view
#

class ContactAssignmentRecord:

    def __init__(self, contact, assignment):
        self.contact = contact
        self.assignment = assignment


class ContactThemeAssignmentsTable(BaseTable):
    """Contact theme contacts table"""

    prefix = 'contacts'
    title = _("Theme contacts assignments")

    cssClasses = {
        'table': 'table table-bordered table-striped table-hover table-tight datatable'
    }

    @reify
    def data_attributes(self):
        result = super().data_attributes
        result['tr']['data-ams-target'] = '_blank'
        return result

    @reify
    def values(self):
        return list(super().values)


@adapter_config(context=(IContactTheme, IPyAMSLayer, ContactThemeAssignmentsTable),
                provides=IValues)
class ContactThemeAssignmentsValues(ContextRequestViewAdapter):
    """Contact theme assignments values adapter"""

    @property
    def values(self):
        contacts = query_utility(IContactManager)
        if contacts is not None:
            yield from [ContactAssignmentRecord(c, a)
                        for c, a in contacts.get_contacts(self.context)]


@adapter_config(context=(ContactAssignmentRecord, IPyAMSLayer, ContactThemeAssignmentsTable),
                provides=ITableElementEditor)
class ContactThemeAssignmentEditorAdapter(ContextRequestViewAdapter):
    """Contact theme assignment editor adapter"""

    @property
    def url(self):
        return absolute_url(self.context.contact, self.request, 'admin')


@adapter_config(name='name',
                context=(IContactTheme, IPyAMSLayer, ContactThemeAssignmentsTable),
                provides=IColumn)
class ContactThemeAssignmentsNameColumn(I18nColumn, GetAttrColumn):
    """Contact theme assignment name column"""

    _header = _("Contact name")
    attrName = 'contact_name'

    weight = 10
    
    def getValue(self, obj):
        return super(ContactThemeAssignmentsNameColumn, self).getValue(obj.contact) or \
            II18n(obj.contact).query_attribute('title', request=self.request) or '--'


@adapter_config(name='email',
                context=(IContactTheme, IPyAMSLayer, ContactThemeAssignmentsTable),
                provides=IColumn)
class ContactThemeAssignmentsEmailColumn(I18nColumn, GetAttrColumn):
    """Contact theme assignment email column"""

    _header = _("Mail address")
    attrName = 'mail_address'

    weight = 20

    def getValue(self, obj):
        return obj.assignment.mail_address or obj.contact.mail_address or '--'


@adapter_config(name='communes',
                context=(IContactTheme, IPyAMSLayer, ContactThemeAssignmentsTable),
                provides=IColumn)
class ContactThemeAssignmentsCommunesColumn(I18nColumn, GetAttrColumn):
    """Contact theme assignment communes column"""

    _header = _("Cities")

    weight = 30

    def getValue(self, obj):
        cities = obj.assignment.cities
        if not cities:
            return '--'
        return '<br />'.join(commune.title for commune in
                             Commune.find_by_insee_code(cities).order_by(Commune.nccenr))


@adapter_config(name='departments',
                context=(IContactTheme, IPyAMSLayer, ContactThemeAssignmentsTable),
                provides=IColumn)
class ContactThemeAssignmentsDepartmentsColumn(I18nColumn, GetAttrColumn):
    """Contact theme assignment departments column"""

    _header = _("Departments")

    weight = 40

    def getValue(self, obj):
        departments = obj.assignment.departments
        if not departments:
            return '--'
        return '<br />'.join(sorted(II18n(department).query_attribute('title', request=self.request)
                                    for department in departments))


@adapter_config(name='structures',
                context=(IContactTheme, IPyAMSLayer, ContactThemeAssignmentsTable),
                provides=IColumn)
class ContactThemeAssignmentsStructuresColumn(I18nColumn, GetAttrColumn):
    """Contact theme assignments structures column"""

    _header = _("Structures")

    weight = 50

    def getValue(self, obj):
        structures = obj.assignment.structures
        if not structures:
            return '--'
        return '<br />'.join(structure.title for structure in
                             Structure.find({'key': structures}).order_by(Structure.code_sign))


@pagelet_config(name='contacts-dialog.html', context=IContactTheme, layer=IPyAMSLayer,
                permission=VIEW_SYSTEM_PERMISSION)
class ContactThemeAssignmentsForm(AdminDialogDisplayForm):
    """Contact theme contacts view"""

    @property
    def title(self):
        return self.request.localizer.translate(_("« {} » theme assignments")).format(
            self.context.get_label(self.request))

    dialog_class = 'modal-max'
    fieldset_class = 'height-600'

    fields = field.Fields(Interface)


@adapter_config(name='contacts', context=(IContactTheme, IPyAMSLayer, ContactThemeAssignmentsForm),
                provides=IInnerSubForm)
class ContactThemeAssignmentsView(InnerTableView):
    """Contact theme contacts view"""

    title = _("Theme assignments")

    table_class = ContactThemeAssignmentsTable
    weight = 100

    hide_switcher = True


@viewlet_config(name='export-to-tsv',
                context=Interface, layer=IAdminLayer,
                view=ContactThemeAssignmentsTable,
                manager=IToolbarViewletManager)
class ContactThemeAssignmentsExportAction(JsToolbarAction):
    """TSV export action"""

    label = _("Export to TSV")
    label_css_class = 'fa fa-fw fa-table'
    css_class = 'btn btn-xs btn-default'

    url = 'MyAMS.container.exportTableToTSV'
