#
# Copyright (c) 2008-2019 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

import calendar

from pyramid.view import view_config
from sqlalchemy import func
from zope.interface import implementer

from onf_website.shared.contact.handler import LeadLog, PARENT_SESSION
from onf_website.shared.contact.interfaces.handler import IContactFormHandlerTarget
from onf_website.shared.contact.zmi.interfaces import IContactFormLeadsMenu
from pyams_alchemy.engine import get_user_session
from pyams_content.interfaces import MANAGE_CONTENT_PERMISSION
from pyams_pagelet.pagelet import pagelet_config
from pyams_sequence.interfaces import ISequentialIdInfo
from pyams_skin.interfaces import IWidgetInnerPage
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.viewlet.menu import MenuItem
from pyams_template.template import template_config
from pyams_viewlet.viewlet import viewlet_config
from pyams_zmi.layer import IAdminLayer
from pyams_zmi.view import InnerAdminView


__docformat__ = 'restructuredtext'

from onf_website import _


@viewlet_config(name='leads-stats.menu', context=IContactFormHandlerTarget, layer=IAdminLayer,
                manager=IContactFormLeadsMenu, permission=MANAGE_CONTENT_PERMISSION, weight=10)
class FormLeadsStatsMenu(MenuItem):
    """Form leads statistics menu"""

    label = _("Leads statistics")
    icon_class = 'fa-bar-chart'
    url = '#leads-stats.html'


@pagelet_config(name='leads-stats.html', context=IContactFormHandlerTarget, layer=IPyAMSLayer,
                permission=MANAGE_CONTENT_PERMISSION)
@template_config(template='templates/leads-stats.pt', layer=IPyAMSLayer)
@implementer(IWidgetInnerPage)
class FormLeadsStatsView(InnerAdminView):
    """Form leads statistics view"""

    title = _("Leads statistics")


@view_config(name='get-leads-stats.json', context=IContactFormHandlerTarget,
             request_type=IPyAMSLayer, permission=MANAGE_CONTENT_PERMISSION,
             renderer='json', xhr=True)
def form_leads_statistics_view(request):
    """Form leads statistics data view"""
    session = get_user_session(PARENT_SESSION)
    data = session.query(func.date_trunc('day', LeadLog.submit_date).label('date'),
                         func.count(LeadLog.id).label('count')) \
        .filter(LeadLog.form_oid == ISequentialIdInfo(request.context).get_short_oid()) \
        .group_by(func.date_trunc('day', LeadLog.submit_date)) \
        .order_by(func.date_trunc('day', LeadLog.submit_date))
    return {
        'settings': {
            'label': "Leads stats",
            'xaxis': {
                'mode': 'time',
                'timeBase': 'milliseconds',
                'monthNames': [calendar.month_abbr[x+1] for x in range(12)],
                'timeformat': '%d %b %Y'
            }
        },
        'data': [{
            'data': [(calendar.timegm(rec.date.timetuple()) * 1000, rec.count)
                     for rec in data],
            'bars': {
                'show': True
            },
            'label': request.localizer.translate(_("Leads/day"))
        }]
    }
