#
# Copyright (c) 2008-2019 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from pyramid.decorator import reify
from pyramid.events import subscriber
from pyramid.view import view_config
from z3c.form import field
from z3c.form.interfaces import IDataExtractedEvent, DISPLAY_MODE
from z3c.table.column import GetAttrColumn
from z3c.table.interfaces import IColumn, IValues
from zope.interface import Invalid, implementer, Interface

from onf_website.shared.contact import IContactAssignment, IContactAssignmentsContainer, IWfContact
from onf_website.shared.contact.assignment import ContactAssignment
from onf_website.shared.contact.interfaces import IWfContactInfo
from onf_website.shared.contact.zmi.interfaces import IContactAssignmentForm
from pyams_content.interfaces import MANAGE_CONTENT_PERMISSION
from pyams_content.shared.common.zmi.properties import SharedContentPropertiesEditForm
from pyams_form.form import AJAXAddForm, ajax_config
from pyams_form.help import FormHelp
from pyams_form.interfaces.form import IInnerSubForm, IFormHelp
from pyams_form.security import ProtectedFormObjectMixin
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.container import switch_element_attribute
from pyams_skin.event import get_json_switched_table_refresh_event, get_json_table_row_refresh_event
from pyams_skin.interfaces.viewlet import IWidgetTitleViewletManager
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.table import AttributeSwitcherColumn, BaseTable, I18nColumn, TrashColumn
from pyams_skin.viewlet.toolbar import ToolbarAction
from pyams_utils.adapter import ContextRequestViewAdapter, adapter_config
from pyams_utils.traversing import get_parent
from pyams_utils.url import absolute_url
from pyams_viewlet.viewlet import viewlet_config
from pyams_zmi.form import AdminDialogAddForm, InnerAdminEditForm, AdminDialogEditForm
from pyams_zmi.interfaces import IPropertiesEditForm
from pyams_zmi.zmi.table import InnerTableView

from onf_website import _


@adapter_config(name='contact-properties',
                context=(IWfContact, IPyAMSLayer, SharedContentPropertiesEditForm),
                provides=IInnerSubForm)
class ContactPropertiesEditForm(InnerAdminEditForm):
    """Contact properties edit form extension"""

    prefix = 'contact_properties.'

    legend = _("Contact properties")
    fieldset_class = 'bordered no-x-margin margin-y-10'

    fields = field.Fields(IWfContactInfo)
    weight = 1


#
# Contact assignments
#

class ContactAssignmentsTable(BaseTable):
    """Contact assignments table"""

    prefix = 'assignments'

    permission = MANAGE_CONTENT_PERMISSION

    hide_header = True
    hide_body_toolbar = True
    sortOn = None

    cssClasses = {'table': 'table table-bordered table-striped table-hover table-tight'}

    @reify
    def data_attributes(self):
        attributes = super().data_attributes
        assignments = self.context.assignments
        if assignments is not None:
            attributes.setdefault('table', {}).update({
                'data-ams-location': absolute_url(assignments, self.request)
            })
            attributes.setdefault('td', {}).update({
                'data-ams-attribute-switcher': self.get_switcher_target,
                'data-ams-switcher-attribute-name': self.get_switcher_attribute
            })
        return attributes

    @staticmethod
    def get_switcher_target(element, column):
        return getattr(column, 'switch_target', None)

    @staticmethod
    def get_switcher_attribute(element, column):
        return getattr(column, 'switch_attribute', None)


@adapter_config(context=(IWfContact, IPyAMSLayer, ContactAssignmentsTable), provides=IValues)
class ContactAssignmentsTableValuesAdapter(ContextRequestViewAdapter):
    """Contact assignments table values adapter"""

    @property
    def values(self):
        return self.context.assignments.values()


@adapter_config(name='enable-disable', context=(IWfContact, IPyAMSLayer, ContactAssignmentsTable), provides=IColumn)
class ContactAssignmentsTableEnablerColumn(AttributeSwitcherColumn):
    """Contact assignments switcher column"""

    switch_target = 'switch-assignment.json'
    switch_attribute = 'active'

    on_icon_class = 'fa fa-fw fa-check-square-o'
    off_icon_class = 'fa fa-fw fa-check-square txt-color-silver opacity-75'

    icon_hint = _("Switch assignment activity")

    weight = 6


@view_config(name='switch-assignment.json', context=IContactAssignmentsContainer,
             request_type=IPyAMSLayer, permission=MANAGE_CONTENT_PERMISSION,
             renderer='json', xhr=True)
def enable_assignment(request):
    """Switch active assignment"""
    return switch_element_attribute(request, IContactAssignmentsContainer, attribute_name='active')


@adapter_config(name='name', context=(IWfContact, IPyAMSLayer, ContactAssignmentsTable),
                provides=IColumn)
class ContactAssignmentsTableNameColumn(I18nColumn, GetAttrColumn):
    """Contacts assignments table name column"""

    _header = _("Contact theme")
    weight = 10

    def getValue(self, obj):
        theme = obj.get_theme()
        if theme is not None:
            return theme.get_label(self.request)


@adapter_config(name='url',
                context=(IWfContact, IPyAMSLayer, ContactAssignmentsTable),
                provides=IColumn)
class ContactAssignmentsTableURLColumn(I18nColumn, GetAttrColumn):
    """Contacts assignments URL column"""

    _header = _("Specific URI")
    attrName = 'target_url'
    weight = 20

    def getValue(self, obj):
        return super(ContactAssignmentsTableURLColumn, self).getValue(obj) or '--'


@adapter_config(name='email',
                context=(IWfContact, IPyAMSLayer, ContactAssignmentsTable),
                provides=IColumn)
class ContactAssignmentsTableEmailColumn(I18nColumn, GetAttrColumn):
    """Contacts assignments email column"""

    _header = _("Specific email")
    attrName = 'mail_address'
    weight = 30

    def getValue(self, obj):
        return super(ContactAssignmentsTableEmailColumn, self).getValue(obj) or '--'


@adapter_config(name='trash',
                context=(IWfContact, IPyAMSLayer, ContactAssignmentsTable),
                provides=IColumn)
class ContactAssignmentsTableTrashColumn(TrashColumn):
    """Contact assignments table trash column"""

    permission = MANAGE_CONTENT_PERMISSION


@adapter_config(name='contact-assignments',
                context=(IWfContact, IPyAMSLayer, IPropertiesEditForm),
                provides=IInnerSubForm)
class ContactAssignmentsView(InnerTableView):
    """Contact assignments view"""

    title = _("Contact assignments")

    table_class = ContactAssignmentsTable
    weight = 10

    @property
    def actions_context(self):
        return IContactAssignmentsContainer(self.context)


#
# Contact assignment add and edit forms
#

@viewlet_config(name='add-contact-assignment.menu', context=IWfContact, layer=IPyAMSLayer,
                view=ContactAssignmentsView, manager=IWidgetTitleViewletManager, weight=10)
class ContactAssignmentAddAction(ProtectedFormObjectMixin, ToolbarAction):
    """Contact assignment add action"""

    label = _("Add assignment")

    url = 'add-contact-assignment.html'
    modal_target = True


@pagelet_config(name='add-contact-assignment.html', context=IContactAssignmentsContainer,
                layer=IPyAMSLayer, permission=MANAGE_CONTENT_PERMISSION)
@ajax_config(name='add-contact-assignment.json', context=IContactAssignmentsContainer,
             layer=IPyAMSLayer, base=AJAXAddForm)
@implementer(IContactAssignmentForm)
class ContactAssignmentAddForm(AdminDialogAddForm):
    """Contact assignment add form"""

    legend = _("Add new contact assignment")
    icon_css_class = 'fa fa-fw fa-tags'

    fields = field.Fields(IContactAssignment).omit('active')
    edit_permission = MANAGE_CONTENT_PERMISSION

    def create(self, data):
        return ContactAssignment()

    def add(self, object):
        self.context[object.theme] = object

    def get_ajax_output(self, changes):
        context = get_parent(self.context, IWfContact)
        view = ContactAssignmentsView(self.context, self.request)
        return {
            'status': 'success',
            'message': self.request.localizer.translate(_("Assignment was correctly added.")),
            'events': [
                get_json_switched_table_refresh_event(context, self.request, view.table_class)
            ]
        }


@subscriber(IDataExtractedEvent, form_selector=ContactAssignmentAddForm)
def extract_contact_assignment_data(event):
    """Extract data from contact assignment add form"""
    container = event.form.context
    if event.data.get('theme') in container:
        event.form.widgets.errors += (Invalid(_("An assignment already exists for this theme!")),)


@pagelet_config(name='properties.html', context=IContactAssignment,
                layer=IPyAMSLayer, permission=MANAGE_CONTENT_PERMISSION)
@ajax_config(name='properties.json', context=IContactAssignment, layer=IPyAMSLayer)
@implementer(IContactAssignmentForm)
class ContactAssignmentPropertiesEditForm(AdminDialogEditForm):
    """Contact assignment properties edit form"""

    legend = _("Edit assignment properties")
    icon_css_class = 'fa fa-fw fa-tags'

    fields = field.Fields(IContactAssignment).omit('active')
    edit_permission = MANAGE_CONTENT_PERMISSION

    def updateWidgets(self, prefix=None):
        super(ContactAssignmentPropertiesEditForm, self).updateWidgets(prefix)
        if 'theme' in self.widgets:
            self.widgets['theme'].mode = DISPLAY_MODE

    def get_ajax_output(self, changes):
        output = super(self.__class__, self).get_ajax_output(changes)
        if changes:
            context = get_parent(self.context, IWfContact)
            view = ContactAssignmentsView(self.context, self.request)
            output.setdefault('events', []).append(
                get_json_table_row_refresh_event(context, self.request, view.table_class, self.context))
        return output


@adapter_config(context=(Interface, IPyAMSLayer, IContactAssignmentForm), provides=IFormHelp)
class ContactAssignmentFormAdapter(FormHelp):
    """Contact assignment form help adapter"""

    message = _("**WARNING**: If you don't define any city, department of structure for this "
                "assignment, this contact will be defined as a *default contact* for this theme "
                "and will receive all notifications about it!\n"
                "\n"
                "You can associate a target URL and an email address to a contact assignment. "
                "If both are defined, target URL is of higher priority; if only email address is "
                "defined, it will be used even if a target URL is affected to the contact "
                "himself.")
    message_format = 'rest'
