#
# Copyright (c) 2008-2019 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from html import escape

from pyramid.decorator import reify
from pyramid.view import view_config
from z3c.form import field
from z3c.table.column import GetAttrColumn
from z3c.table.interfaces import IColumn, IValues
from zope.interface import implementer

from onf_website.shared.contact.handler import LeadLog, PARENT_SESSION
from onf_website.shared.contact.interfaces.handler import IContactFormHandlerTarget, ILeadLog
from onf_website.shared.contact.zmi.interfaces import IContactFormLeadsMenu
from pyams_alchemy.engine import get_user_session
from pyams_content.interfaces import MANAGE_CONTENT_PERMISSION
from pyams_content.shared.form import IFormFieldContainer
from pyams_i18n.interfaces import II18n
from pyams_pagelet.pagelet import pagelet_config
from pyams_sequence.interfaces import ISequentialIdInfo
from pyams_skin.container import ContainerView
from pyams_skin.interfaces.container import ITableElementEditor
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.table import BaseTable, I18nColumn, NameColumn, TrashColumn
from pyams_skin.viewlet.menu import MenuItem
from pyams_utils.adapter import ContextRequestViewAdapter, adapter_config
from pyams_utils.date import SH_DATETIME_FORMAT, format_datetime
from pyams_utils.list import boolean_iter
from pyams_utils.schema import IJSONDictFieldsGetter
from pyams_utils.url import absolute_url
from pyams_viewlet.manager import viewletmanager_config
from pyams_viewlet.viewlet import viewlet_config
from pyams_workflow.interfaces import IWorkflowVersions
from pyams_zmi.form import AdminDialogDisplayForm
from pyams_zmi.interfaces.menu import IContentManagementMenu
from pyams_zmi.layer import IAdminLayer
from pyams_zmi.view import InnerAdminView


__docformat__ = 'restructuredtext'

from onf_website import _


@viewlet_config(name='form-leads.menu', context=IContactFormHandlerTarget, layer=IAdminLayer,
                manager=IContentManagementMenu, permission=MANAGE_CONTENT_PERMISSION, weight=20)
@viewletmanager_config(name='form-leads.menu', layer=IAdminLayer, provides=IContactFormLeadsMenu)
@implementer(IContactFormLeadsMenu)
class ContactFormLeadsMenu(MenuItem):
    """Contact form leads menu"""

    label = _("Leads")
    icon_class = 'fa-vcard'
    url = '#form-leads.html'


class FormLeadsTable(BaseTable):
    """Form leads table"""

    prefix = 'leads'
    title = _("Contact leads")

    @property
    def cssClasses(self):
        classes = ['table', 'table-bordered', 'table-striped', 'table-hover',
                   'table-tight', 'datatable']
        return {'table': ' '.join(classes)}

    @reify
    def data_attributes(self):
        attributes = super().data_attributes
        attributes.setdefault('table', {}).update({
            'data-ams-delete-target': 'delete-lead.json',
            'data-ams-datatable-sorting': '1,desc'
        })
        attributes.setdefault('tr', {}).update({
            'data-ams-element-name': lambda x, col: x.id
        })
        return attributes

    def render(self):
        has_values, values = boolean_iter(self.values)
        if not has_values:
            translate = self.request.localizer.translate
            return translate(_("No currently available lead."))
        return super().render()


@adapter_config(context=(IContactFormHandlerTarget, IPyAMSLayer, FormLeadsTable),
                provides=IValues)
class FormLeadsValues(ContextRequestViewAdapter):
    """Form leads table values adapter"""

    @property
    def values(self):
        session = get_user_session(PARENT_SESSION)
        sequence = ISequentialIdInfo(self.context)
        return session.query(LeadLog) \
            .filter(LeadLog.form_oid == sequence.get_short_oid()) \
            .order_by(LeadLog.submit_date.desc())


@adapter_config(name='oid',
                context=(IContactFormHandlerTarget, IPyAMSLayer, FormLeadsTable),
                provides=IColumn)
class FormLeadsTableOIDColumn(I18nColumn, GetAttrColumn):
    """Form leads table OID column"""

    _header = _("Référence")
    attrName = 'oid'
    weight = 8


@adapter_config(name='submit-date',
                context=(IContactFormHandlerTarget, IPyAMSLayer, FormLeadsTable),
                provides=IColumn)
class FormLeadsTableDateColumn(I18nColumn, GetAttrColumn):
    """Form leads table date column"""

    _header = _("Date")
    attrName = 'submit_date'
    weight = 10

    def getValue(self, obj):
        value = super(FormLeadsTableDateColumn, self).getValue(obj)
        if value:
            return format_datetime(value, SH_DATETIME_FORMAT)
        return '--'


@adapter_config(name='name',
                context=(IContactFormHandlerTarget, IPyAMSLayer, FormLeadsTable),
                provides=IColumn)
class FormLeadsTableNameColumn(NameColumn):
    """Form leads table name column"""

    _header = _("Email")
    attrName = 'submit_email'
    weight = 20

    def getValue(self, obj):
        return super(FormLeadsTableNameColumn, self).getValue(obj) or '--'


@adapter_config(name='contacts',
                context=(IContactFormHandlerTarget, IPyAMSLayer, FormLeadsTable),
                provides=IColumn)
class FormLeadsTableContactsColumn(I18nColumn, GetAttrColumn):
    """Form leads table contacts column"""

    _header = _("Contacts")
    attrName = 'contacts'
    weight = 30

    def getValue(self, obj):
        value = super(FormLeadsTableContactsColumn, self).getValue(obj)
        if value:
            value = escape(value)
        return value or '--'


@adapter_config(name='trash',
                context=(IContactFormHandlerTarget, IPyAMSLayer, FormLeadsTable),
                provides=IColumn)
class FormLeadsTableTrashColumn(TrashColumn):
    """Form leads table trash column"""

    permission = None


@view_config(name='delete-lead.json', context=IContactFormHandlerTarget, request_type=IPyAMSLayer,
             permission=MANAGE_CONTENT_PERMISSION, renderer='json', xhr=True)
def delete_form_lead(request):
    """Form lead delete view"""
    translate = request.localizer.translate
    name = request.params.get('object_name')
    if not name:
        return {
            'status': 'message',
            'messagebox': {
                'status': 'error',
                'content': translate(_("No provided object_name argument!"))
            }
        }
    try:
        name = int(name)
    except ValueError:
        return {
            'status': 'message',
            'messagebox': {
                'status': 'error',
                'content': translate(_("Given element name is not a valid ID!"))
            }
        }
    session = get_user_session(PARENT_SESSION)
    lead = session.query(LeadLog) \
        .get(name)
    if lead is None:
        return {
            'status': 'message',
            'messagebox': {
                'status': 'error',
                'content': translate(_("Provided element name doesn't match any valid lead!"))
            }
        }
    session.delete(lead)
    return {'status': 'success'}


@pagelet_config(name='form-leads.html', context=IContactFormHandlerTarget, layer=IPyAMSLayer,
                permission=MANAGE_CONTENT_PERMISSION)
class FormLeadsView(InnerAdminView, ContainerView):
    """Form leads view"""

    table_class = FormLeadsTable


#
# Form lead details
#

@adapter_config(context=(LeadLog, IPyAMSLayer, FormLeadsTable), provides=ITableElementEditor)
class FormLeadLogElementEditor(ContextRequestViewAdapter):
    """Form lead log element editor adapter"""

    modal_target = True

    @property
    def url(self):
        return absolute_url(self.view.context, self.request, 'show-lead.html',
                            {'id': self.context.id})


@pagelet_config(name='show-lead.html', context=IContactFormHandlerTarget, layer=IPyAMSLayer,
                permission=MANAGE_CONTENT_PERMISSION)
class FormLeadLogView(AdminDialogDisplayForm):
    """Form lead log view"""

    legend = _("Lead details")
    dialog_class = 'modal-large'
    icon_css_class = 'fa fa-fw fa-vcard'

    fields = field.Fields(ILeadLog)

    def getContent(self):
        session = get_user_session(PARENT_SESSION)
        id = self.request.params.get('id')
        return session.query(LeadLog).get(int(id))


@adapter_config(context=(ILeadLog, IPyAMSLayer, FormLeadLogView), provides=IJSONDictFieldsGetter)
class WfFormFieldsGetter(ContextRequestViewAdapter):
    """Form fields getter"""

    def get_fields(self, data):
        form = IWorkflowVersions(self.view.context).get_version(self.context.form_version)
        if form is None:
            yield from map(lambda item: (item[0], item[0], item[1]), data)
        else:
            for name, field in IFormFieldContainer(form).items():
                label = II18n(field).query_attribute('label', request=self.request)
                yield name, label, data.get(name) or '--'
