#
# Copyright (c) 2008-2019 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from persistent import Persistent
from zope.container.contained import Contained
from zope.container.folder import Folder
from zope.interface import implementer
from zope.schema.fieldproperty import FieldProperty
from zope.schema.vocabulary import SimpleTerm, SimpleVocabulary
from zope.traversing.interfaces import ITraversable

from onf_website.shared.contact.interfaces import IContactManager
from onf_website.shared.contact.interfaces.theme import CONTACT_THEMES_VOCABULARY, \
    CONTACT_THEME_MANAGER_ANNOTATION_KEY, IContactTheme, IContactThemeManager, \
    IContactThemeManagerTarget
from onf_website.skin.public.layer import IONFBaseLayer
from pyams_content.component.extfile.interfaces import IExtFileContainerTarget
from pyams_content.component.links import InternalReferenceMixin
from pyams_content.component.links.interfaces import ILinkContainerTarget
from pyams_content.component.paragraph import IParagraphContainerTarget
from pyams_content.interfaces import MANAGE_TOOL_PERMISSION
from pyams_form.interfaces.form import IFormContextPermissionChecker
from pyams_i18n.interfaces import II18n
from pyams_utils.adapter import ContextAdapter, adapter_config, get_annotation_adapter
from pyams_utils.factory import factory_config
from pyams_utils.registry import query_utility
from pyams_utils.request import query_request
from pyams_utils.vocabulary import vocabulary_config


__docformat__ = 'restructuredtext'


@implementer(IContactTheme, IParagraphContainerTarget, IExtFileContainerTarget,
             ILinkContainerTarget)
class ContactTheme(InternalReferenceMixin, Persistent, Contained):
    """Contact theme persistent class"""

    name = FieldProperty(IContactTheme['name'])
    internal_label = FieldProperty(IContactTheme['internal_label'])
    label = FieldProperty(IContactTheme['label'])
    crm_category = FieldProperty(IContactTheme['crm_category'])
    notepad = FieldProperty(IContactTheme['notepad'])

    header = FieldProperty(IContactTheme['header'])
    teaser = FieldProperty(IContactTheme['teaser'])
    reference = FieldProperty(IContactTheme['reference'])

    def get_label(self, request):
        if IONFBaseLayer.providedBy(request):
            return II18n(self).query_attribute('label', request=request) or self.name
        return self.internal_label or self.name


@adapter_config(context=IContactTheme, provides=IFormContextPermissionChecker)
class ContactThemePermissionChecker(ContextAdapter):
    """Contact theme permission checker"""

    edit_permission = MANAGE_TOOL_PERMISSION


@factory_config(IContactThemeManager)
class ContactThemeManager(Folder):
    """Contact themes manager persistent class"""


@adapter_config(context=IContactThemeManagerTarget, provides=IContactThemeManager)
def contact_theme_manager_factory(context):
    """Contact themes manager factory"""
    return get_annotation_adapter(context, CONTACT_THEME_MANAGER_ANNOTATION_KEY,
                                  IContactThemeManager, name='++themes++')


@adapter_config(name='themes', context=IContactThemeManagerTarget, provides=ITraversable)
class ContactThemeManagerThemesTraverser(ContextAdapter):
    """Contact themes manager ++themes++ namespace traverser"""

    def traverse(self, name, furtherpath=None):
        return IContactThemeManager(self.context)


@vocabulary_config(name=CONTACT_THEMES_VOCABULARY)
class ContactThemesVocabulary(SimpleVocabulary):
    """Contact themes vocabulary"""

    def __init__(self, context=None):
        terms = []
        contacts = query_utility(IContactManager)
        if contacts is not None:
            manager = IContactThemeManager(contacts)
            request = query_request()
            for (key, theme) in manager.items():
                terms.append(SimpleTerm(key, title=theme.get_label(request)))
        super(ContactThemesVocabulary, self).__init__(terms)
