#
# Copyright (c) 2008-2019 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#
from z3c.form.browser.checkbox import CheckBoxWidget
from z3c.form.interfaces import HIDDEN_MODE, ICheckBoxWidget, IFieldWidget, INPUT_MODE, NO_VALUE
from z3c.form.widget import FieldWidget
from zope.contentprovider.interfaces import IContentProvider
from zope.interface import implementer_only
from zope.publisher.interfaces import IPublisherRequest

from onf_website.shared.contact import IContactManager
from onf_website.shared.contact.field import IContactThemeField, IContactThemesListField
from onf_website.shared.contact.interfaces.field import ALWAYS_DISPLAY_MODE, HIDDEN_DISPLAY_MODE, \
    HIDE_PRESELECTED_DISPLAY_MODE, IContactThemeFieldInfo, MULTIPLE_DISPLAY_MODE
from onf_website.shared.contact.interfaces.theme import IContactThemeManager
from onf_website.skin.public.layer import IONFBaseLayer
from pyams_content.shared.form import IFormFieldContainer, IWfForm
from pyams_form.interfaces import IFormLayer
from pyams_form.interfaces.form import ISelect2Widget
from pyams_form.widget import Select2Widget, override_widget_template
from pyams_utils.adapter import adapter_config
from pyams_utils.interfaces.data import IObjectData
from pyams_utils.registry import get_utility
from pyams_utils.url import absolute_url


__docformat__ = 'restructuredtext'


#
# Simple contact theme selection widget
#

class IContactThemeSelectWidget(ISelect2Widget):
    """Contact theme select widget"""


@implementer_only(IObjectData, IContactThemeSelectWidget)
class ContactThemeSelectWidget(Select2Widget):
    """Contact theme widget"""

    theme = None
    object_data = {}

    @property
    def notice_id(self):
        return '{0}_footer'.format(self.id)

    @property
    def noValueMessage(self):
        field = IFormFieldContainer(self.context).get(self.field.__name__)
        return field.placeholder or super(ContactThemeSelectWidget, self).noValueMessage

    def update(self):
        super(ContactThemeSelectWidget, self).update()
        # check selected theme
        request_theme = None
        contacts = get_utility(IContactManager)
        if contacts is not None:
            themes = IContactThemeManager(contacts)
            request_theme = theme = self.request.params.get('theme')
            if (theme is None) and (len(self.items) == 1):  # only one selected theme
                theme = self.items[0].get('value')
            if theme is not None:
                self.value = theme
                self.theme = themes.get(theme)
            self.object_data = {
                'ams-change-handler': 'MyAMS.helpers.select2ChangeHelper',
                'stop-propagation': True,
                'ams-select2-helper-type': 'html',
                'ams-select2-helper-url': absolute_url(themes, self.request,
                                                       'get-theme-infos.html'),
                'ams-select2-helper-target': '#{0}'.format(self.notice_id)
            }
        # get and check field mode
        field = IFormFieldContainer(self.context).get(self.field.__name__)
        field_info = IContactThemeFieldInfo(field, None)
        mode = field_info.display_mode if field_info is not None else ALWAYS_DISPLAY_MODE
        if mode == HIDDEN_DISPLAY_MODE:
            self.mode = HIDDEN_MODE
        elif mode == MULTIPLE_DISPLAY_MODE:
            if len(self.items) < 2:
                self.mode = HIDDEN_MODE
        elif mode == HIDE_PRESELECTED_DISPLAY_MODE:
            if request_theme is not None:
                self.mode = HIDDEN_MODE

    @property
    def after_widget_notice(self):
        notice = ''
        if self.theme is not None:
            registry = self.request.registry
            if IPublisherRequest.providedBy(self.request):
                request = self.request._request
            else:
                request = self.request
            content = registry.queryMultiAdapter((self.theme, request), IContentProvider,
                                                 name='theme-info')
            if content is not None:
                content.update()
                notice = content.render()
        return '<span id="{0}" class="theme-info">{1}</span>'.format(self.notice_id, notice)


@adapter_config(context=(IContactThemeField, IFormLayer), provides=IFieldWidget)
def ContactThemeFieldWidget(field, request):
    return FieldWidget(field, ContactThemeSelectWidget(request))


#
# Multiple contact themes list selection widget
#

class IContactThemesListSelectWidget(ICheckBoxWidget):
    """Contact themes list widget"""


@implementer_only(IContactThemesListSelectWidget)
class ContactThemesListSelectWidget(CheckBoxWidget):
    """Contact themes list select widget"""

    def extract(self, default=NO_VALUE):
        value = self.request.get('theme', default)
        if value != default:
            if not isinstance(value, (tuple, list)):
                # this is here to make any single value a tuple
                value = (value,)
            if not isinstance(value, tuple):
                # this is here to make a non-tuple (just a list at this point?)
                # a tuple. the dance is about making return values uniform
                value = tuple(value)
            # do some kind of validation, at least only use existing values
            for token in value:
                if token == self.noValueToken:
                    continue
                try:
                    self.terms.getTermByToken(token)
                except LookupError:
                    return default
            return value
        return super().extract(default)


override_widget_template(widget=IContactThemesListSelectWidget,
                         template='templates/themes-list-input.pt',
                         context=IWfForm,
                         layer=IONFBaseLayer,
                         mode=INPUT_MODE)


@adapter_config(context=(IContactThemesListField, IFormLayer),
                provides=IFieldWidget)
def ContactThemesListFieldWidget(field, request):
    return FieldWidget(field, ContactThemesListSelectWidget(request))
