#
# Copyright (c) 2015-2023 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS_*** module

"""

from colander import Float, Integer, MappingSchema, SchemaNode, SequenceSchema, String, drop
from cornice import Service
from cornice.validators import colander_validator
from pyramid.httpexceptions import HTTPOk

from onf_website.component.location.api.schema import Location
from onf_website.shared.common.interfaces import REST_CONTENT_PUBLIC_GETTER_ROUTE, REST_CONTENT_PUBLIC_SEARCH_ROUTE
from pyams_content.shared.common.api import SharedContentInfo as BaseSharedContent, content_getter_service_get, \
    content_search_service_get
from pyams_content.shared.common.api.schema import ContentSearchQuery as BaseContentSearchQuery
from pyams_gis.interfaces import WGS84
from pyams_security.rest import check_cors_origin, set_cors_headers
from pyams_utils.rest import BaseResponseSchema, rest_responses

__docformat__ = 'restructuredtext'


#
# Generic content search service
#

content_search_service = Service(name=REST_CONTENT_PUBLIC_SEARCH_ROUTE,
                                 pyramid_route=REST_CONTENT_PUBLIC_SEARCH_ROUTE,
                                 description="Shared content search service")


@content_search_service.options(validators=(check_cors_origin, set_cors_headers))
def onf_content_search_service_options(request):
    return ''


class BaseLocationSearchQuery(BaseContentSearchQuery):
    """Base location search query"""
    srid = SchemaNode(Integer(),
                      description="EPSG spatial reference ID used for location coordinates; default is 4326 for WGS84",
                      missing=WGS84)
    id_nat_frt = SchemaNode(String(),
                            description="List of forest location IDs",
                            missing=drop)


class ContentSearchQuery(BaseLocationSearchQuery):
    longitude = SchemaNode(Float(),
                           description="Origin location longitude, using provided SRID coordinates system",
                           missing=drop)
    latitude = SchemaNode(Float(),
                          description="Origin location latitude, using provided SRID coordinates system",
                          missing=drop)
    distance = SchemaNode(Float(),
                          description="Distance from location origin, in kilometers",
                          missing=50.)


class SharedContent(BaseSharedContent):
    """Shared content schema"""
    location = Location(description="Content location")


class ContentResultsList(SequenceSchema):
    """Content results list"""
    content = SharedContent()


class ContentSearchResult(BaseResponseSchema):
    """Content search results"""
    results = ContentResultsList(description="Content search results",
                                 missing=drop)


class ContentSearchRequest(MappingSchema):
    """Content search request schema"""
    querystring = ContentSearchQuery()


class ContentSearchResponse(MappingSchema):
    """Content search response"""
    body = ContentSearchResult()


content_search_responses = rest_responses.copy()
content_search_responses[HTTPOk.code] = ContentSearchResponse(description="Content search results")


@content_search_service.get(schema=ContentSearchRequest(),
                            validators=(check_cors_origin, colander_validator, set_cors_headers),
                            response_schemas=content_search_responses)
def onf_content_search_service_getter(request):
    return content_search_service_get(request)


#
# Generic content getter service
#

content_getter_service = Service(name=REST_CONTENT_PUBLIC_GETTER_ROUTE,
                                 pyramid_route=REST_CONTENT_PUBLIC_GETTER_ROUTE,
                                 description="Shared content getter service")


@content_getter_service.options(validators=(check_cors_origin, set_cors_headers))
def onf_content_getter_service_options(request):
    return ''


class ContentGetterRequest(MappingSchema):
    """Content getter request"""
    querystring = BaseLocationSearchQuery()


class ContentGetterResult(BaseResponseSchema):
    """Content getter result"""
    content = SharedContent(description="Shared content properties")


class ContentGetterResponse(MappingSchema):
    """Content getter response"""
    body = ContentGetterResult()


content_getter_responses = rest_responses.copy()
content_getter_responses[HTTPOk.code] = ContentGetterResponse(description="Content getter result")


@content_getter_service.get(schema=ContentGetterRequest(),
                            validators=(check_cors_origin, colander_validator, set_cors_headers),
                            response_schemas=content_getter_responses)
def onf_content_getter_service_get(request):
    return content_getter_service_get(request)
