#
# Copyright (c) 2015-2021 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS_*** module

"""

from zope.interface import Interface

from onf_website.component.location import ILocationInfo
from onf_website.shared.activity import IActivityInfo, IWfActivity
from onf_website.shared.activity.interfaces import IActivityInfoRenderer, IActivityManager
from onf_website.shared.common.skin.head import BaseHeaderPortletRenderer
from onf_website.skin.public import baguettebox
from onf_website.skin.public.layer import IONFBaseLayer
from pyams_content.component.gallery import IBaseGallery
from pyams_content.component.theme import IThemesInfo
from pyams_content.features.renderer import IFakeVocabularyContent
from pyams_content.shared.common.portlet.interfaces import ISharedContentHeaderPortletSettings
from pyams_i18n.interfaces import II18n
from pyams_portal.interfaces import IPortalContext, IPortletRenderer
from pyams_skin.layer import IPyAMSLayer
from pyams_template.template import template_config
from pyams_utils.adapter import adapter_config
from pyams_utils.registry import query_utility
from pyams_viewlet.viewlet import ViewContentProvider

__docformat__ = 'restructuredtext'

from onf_website import _


class BaseActivityInfoRenderer(ViewContentProvider):
    """Base activity info renderer"""

    def __init__(self, context, request, view=None):
        super().__init__(context, request, view)


@adapter_config(name='datatype',
                context=(IWfActivity, IPyAMSLayer),
                provides=IActivityInfoRenderer)
@template_config(template='templates/info-datatype.pt', layer=IONFBaseLayer)
class ActivityDatatypeInfoRenderer(BaseActivityInfoRenderer):
    """Activity datatype info renderer"""

    label = _("Activity type")
    weight = 10


@adapter_config(name='location',
                context=(IWfActivity, IPyAMSLayer),
                provides=IActivityInfoRenderer)
@template_config(template='templates/info-location.pt', layer=IONFBaseLayer)
class ActivityLocationInfoRenderer(BaseActivityInfoRenderer):
    """Activity location info renderer"""

    label = _("Location")
    weight = 20

    @property
    def location(self):
        location = ILocationInfo(self.context, None)
        if location is None:
            return
        yield from location.get_elements()


@adapter_config(name='price',
                context=(IWfActivity, IPyAMSLayer),
                provides=IActivityInfoRenderer)
@template_config(template='templates/info-price.pt', layer=IONFBaseLayer)
class ActivityPriceInfoRenderer(BaseActivityInfoRenderer):
    """Activity price info renderer"""

    label = _("Price")
    weight = 30

    @property
    def price(self):
        datatype = self.context.get_data_type()
        if 'price' not in datatype.field_names:
            return None
        info = IActivityInfo(self.context)
        return info.price


@adapter_config(name='adaptations',
                context=(IWfActivity, IPyAMSLayer),
                provides=IActivityInfoRenderer)
@template_config(template='templates/info-adaptations.pt', layer=IONFBaseLayer)
class ActivityAdaptationsInfoRenderer(BaseActivityInfoRenderer):
    """Activity adaptations info renderer"""

    def __new__(cls, context, request):
        if (not IFakeVocabularyContent.providedBy(context)) and (not IThemesInfo(context).themes):
            return None
        return BaseActivityInfoRenderer.__new__(cls)

    label = _("Adaptations")
    weight = 40

    @property
    def adaptations(self):
        manager = query_utility(IActivityManager)
        if manager is None:
            return None
        if manager.adaptations_themes is None:
            return None
        child_themes = manager.adaptations_themes.specifics
        return (
            (theme.alt or theme.label)
            for theme in sorted([
                theme
                for theme in (IThemesInfo(self.context).themes or ())
                if theme in child_themes
            ], key=lambda x: (x.order or 999, x.alt or x.label))
        )


@adapter_config(name='markup',
                context=(IWfActivity, IPyAMSLayer),
                provides=IActivityInfoRenderer)
@template_config(template='templates/info-markup.pt', layer=IONFBaseLayer)
class ActivityMarkupInfoRenderer(BaseActivityInfoRenderer):
    """Activity price info renderer"""

    label = _("Markup")
    weight = 50

    @property
    def markup(self):
        datatype = self.context.get_data_type()
        if 'markup' not in datatype.field_names:
            return None
        info = IActivityInfo(self.context)
        if info.markup is True:
            return self.request.localizer.translate(_('Yes'))
        if info.markup is False:
            return self.request.localizer.translate(_('No'))
        return None


@adapter_config(name='duration',
                context=(IWfActivity, IPyAMSLayer),
                provides=IActivityInfoRenderer)
@template_config(template='templates/info-duration.pt', layer=IONFBaseLayer)
class ActivityDurationInfoRenderer(BaseActivityInfoRenderer):
    """Activity duration info renderer"""

    label = _("Duration")
    weight = 60

    @property
    def duration(self):
        datatype = self.context.get_data_type()
        if 'duration' not in datatype.field_names:
            return None
        info = IActivityInfo(self.context)
        return II18n(info).query_attribute('duration', request=self.request)


@adapter_config(name='length',
                context=(IWfActivity, IPyAMSLayer),
                provides=IActivityInfoRenderer)
@template_config(template='templates/info-length.pt', layer=IONFBaseLayer)
class ActivityLengthInfoRenderer(BaseActivityInfoRenderer):
    """Activity length info renderer"""

    label = _("Length")
    weight = 70

    @property
    def length(self):
        datatype = self.context.get_data_type()
        if 'length' not in datatype.field_names:
            return None
        info = IActivityInfo(self.context)
        return II18n(info).query_attribute('length', request=self.request)


@adapter_config(name='altitude',
                context=(IWfActivity, IPyAMSLayer),
                provides=IActivityInfoRenderer)
@template_config(template='templates/info-altitude.pt', layer=IONFBaseLayer)
class ActivityAltitudeInfoRenderer(BaseActivityInfoRenderer):
    """Activity altitude info renderer"""

    label = _("Altitude")
    weight = 80

    @property
    def altitude(self):
        datatype = self.context.get_data_type()
        if 'altitude' not in datatype.field_names:
            return None
        info = IActivityInfo(self.context)
        return info.altitude


#
# Main head portlet renderer
#


@adapter_config(name='ONF::head-activity',
                context=(IPortalContext, IPyAMSLayer, Interface,
                         ISharedContentHeaderPortletSettings),
                provides=IPortletRenderer)
@template_config(template='templates/head-activities.pt', layer=IONFBaseLayer)
class ActivityHeaderPortletRenderer(BaseHeaderPortletRenderer):
    """Activity header portlet renderer"""

    label = _("ONF: Activity header will images gallery")
    weight = 60

    resources = (baguettebox, )

    def get_medias(self):
        gallery = IBaseGallery(self.context, None)
        if gallery is not None:
            yield from gallery.get_visible_medias()

    @staticmethod
    def get_fixed_infos_count():
        manager = query_utility(IActivityManager)
        if manager is not None:
            return manager.fixed_renderers_count

    def get_renderers(self):
        activity = IWfActivity(self.context, None)
        if activity is None:
            return
        manager = query_utility(IActivityManager)
        if manager is None:
            return
        request = self.request
        registry = request.registry
        for renderer_name in manager.infos_renderers:
            renderer = registry.queryMultiAdapter((activity, request), IActivityInfoRenderer,
                                                  name=renderer_name)
            if renderer is not None:
                renderer.update()
                result = renderer.render().strip()
                if result:
                    yield result
