#
# Copyright (c) 2015-2021 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS_*** module

"""
from collections import OrderedDict

from zope.interface import Interface
from zope.schema import Bool, Choice, Int, List, TextLine
from zope.schema.vocabulary import SimpleTerm, SimpleVocabulary

from pyams_content.features.renderer import IContentRenderer
from pyams_content.shared.common import ISharedContent
from pyams_content.shared.common.interfaces import ISharedToolPortalContext, \
    IWfSharedContentPortalContext
from pyams_i18n.schema import I18nTextLineField
from pyams_sequence.interfaces import IInternalReferencesList
from pyams_sequence.schema import InternalReferenceField
from pyams_thesaurus.schema import ThesaurusTermField


__docformat__ = 'restructuredtext'

from onf_website import _


ACTIVITY_CONTENT_TYPE = 'activity'
ACTIVITY_CONTENT_NAME = _("Activity")

ACTIVITY_INFO_RENDERERS_VOCABULARY = 'onf_website.activity.info.renderers'


class IActivityManager(ISharedToolPortalContext):
    """Activity manager"""

    adaptations_themes = ThesaurusTermField(title=_("Adaptations themes"),
                                            description=_("Select parent theme of themes used to "
                                                          "describe adapted accesses"),
                                            required=False)

    foretmapper_adaptations_themes = ThesaurusTermField(title=_("ForetMapper adaptations themes"),
                                                        description=_("Select parent theme of themes used to "
                                                                      "describe adapted accesses on ForetMapper "
                                                                      "mobile application"),
                                                        required=False)

    default_hearing_source = Choice(title=_("Default origin site"),
                                    description=_("This site will be used as source site for all "
                                                  "forests contents"),
                                    required=False,
                                    vocabulary='PyAMS shared sites')

    default_source_folder = InternalReferenceField(
        title=_("Default source folder"),
        description=_("Main folder to which content is attached; if this reference "
                      "is set, canonical URL will be based on this object's URL for all forests"),
        required=False)

    infos_renderers = List(title=_("Informations renderers"),
                           description=_("Ordered list of selected informations renderers"),
                           value_type=Choice(vocabulary=ACTIVITY_INFO_RENDERERS_VOCABULARY),
                           required=False)

    fixed_renderers_count = Int(title=_("Fixed renderers count"),
                                description=_("Informations renderer past this counter will "
                                              "only be visible by using a \"See more\" button..."),
                                required=True,
                                default=6)


class IActivityManagerFactory(Interface):
    """Activity manager factory interface"""


class IWfActivity(IWfSharedContentPortalContext, IInternalReferencesList):
    """Activity content interface"""


class IWfActivityFactory(Interface):
    """Activity content factory interface"""


class IActivity(ISharedContent):
    """Workflow managed activity interface"""


ACTIVITY_INFO_ANNOTATIONS_KEY = 'onf_website.activity'

ACTIVITY_MARKUP_VALUES = OrderedDict((
    (True, _("Yes")),
    (False, _("No")),
    ('', _("Not relevant"))
))

ACTIVITY_MARKUP_VOCABULARY = SimpleVocabulary([
    SimpleTerm(v, str(v), t)
    for v, t in ACTIVITY_MARKUP_VALUES.items()
])


class IActivityInfo(Interface):
    """Activity info interface"""

    markup = Choice(title=_("Activity markups?"),
                    description=_("Does the activity provides markups?"),
                    required=False,
                    vocabulary=ACTIVITY_MARKUP_VOCABULARY,
                    default=False)

    duration = I18nTextLineField(title=_("Duration"),
                                 description=_("Estimation of activity duration"),
                                 required=False)

    length = I18nTextLineField(title=_("Length"),
                               description=_("Activity length"),
                               required=False)

    altitude = Int(title=_("Altitude"),
                   description=_("Activity altitude, in meters"),
                   required=False)

    price = TextLine(title=_("Price"),
                     description=_("Activity price"),
                     required=False)


class IActivityInfoRenderer(IContentRenderer):
    """Activity information renderer interface"""
