#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'


# import standard library

# import interfaces
from onf_website.reference.target.interfaces import ITargetTable, ITarget
from pyams_i18n.interfaces import II18n, II18nManager
from zope.component.interfaces import ISite
from zope.lifecycleevent.interfaces import IObjectAddedEvent

# import packages
from pyams_content.reference import ReferenceInfo
from pyams_utils.container import BTreeOrderedContainer
from pyams_utils.registry import query_utility
from pyams_utils.request import check_request
from pyams_utils.traversing import get_parent
from pyams_utils.vocabulary import vocabulary_config
from pyramid.events import subscriber
from zope.interface import implementer
from zope.schema.fieldproperty import FieldProperty
from zope.schema.vocabulary import SimpleVocabulary, SimpleTerm


@implementer(ITargetTable, II18nManager)
class TargetTable(BTreeOrderedContainer):
    """Targets table"""

    title = FieldProperty(ITargetTable['title'])
    short_name = FieldProperty(ITargetTable['short_name'])

    languages = FieldProperty(II18nManager['languages'])


@subscriber(IObjectAddedEvent, context_selector=ITargetTable)
def handle_added_target_table(event):
    """Handle new target table"""
    site = get_parent(event.object, ISite)
    registry = site.getSiteManager()
    if registry is not None:
        registry.registerUtility(event.object, ITargetTable)


@implementer(ITarget)
class Target(ReferenceInfo):
    """Target record"""

    facets_label = FieldProperty(ITarget['facets_label'])


@vocabulary_config(name='ONF targets')
class TargetVocabulary(SimpleVocabulary):
    """Targets vocabulary"""

    def __init__(self, context=None):
        table = query_utility(ITargetTable)
        if table is not None:
            request = check_request()
            terms = [SimpleTerm(v,
                                token=v.__name__,
                                title=II18n(v).query_attribute('title', request=request))
                     for v in table.values()]
        else:
            terms = []
        super(TargetVocabulary, self).__init__(terms)
