
import json

from z3c.form.browser.widget import HTMLFormElement, HTMLInputWidget
from z3c.form.converter import BaseDataConverter
from z3c.form.interfaces import DISPLAY_MODE, IDataConverter, IFieldWidget, INPUT_MODE, NO_VALUE
from z3c.form.widget import FieldWidget, MultiWidget, Widget
from zope.interface import implementer_only
from zope.schema.vocabulary import getVocabularyRegistry

from onf_website.reference.forest.interfaces import FOREST_OWNERS_CATEGORIES
from onf_website.reference.planning.model.planning import PlanningData
from onf_website.reference.planning.schema import IPlanningField, IPlanningsListField
from onf_website.reference.planning.zmi.interfaces import IPlanningWidget, IPlanningsListWidget
from pyams_form.interfaces.form import IFormLayer
from pyams_form.widget import widgettemplate_config
from pyams_utils.adapter import adapter_config


__docformat__ = 'restructuredtext'


#
# Planning field
#

@adapter_config(context=(IPlanningField, IPlanningWidget), provides=IDataConverter)
class PlanningDataConverter(BaseDataConverter):
    """Planning data converter"""

    def toWidgetValue(self, value):
        if not value:
            return None
        return PlanningData.find_by_id(value).first()

    def toFieldValue(self, value):
        return value


@widgettemplate_config(mode='input', template='templates/planning-input.pt', layer=IFormLayer)
@widgettemplate_config(mode='display', template='templates/planning-display.pt', layer=IFormLayer)
@implementer_only(IPlanningWidget)
class PlanningWidget(HTMLInputWidget, Widget):
    """Planning widget"""

    @property
    def value_map(self):
        if not self.value:
            return ''
        return json.dumps({self.value.token: self.value.title})


@adapter_config(context=(IPlanningField, IFormLayer), provides=IFieldWidget)
def PlanningFieldWidget(field, request):
    """Planning field widget factory"""
    return FieldWidget(field, PlanningWidget(request))


#
# Plannings list field
#

@adapter_config(context=(IPlanningsListField, IPlanningsListWidget), provides=IDataConverter)
class PlanningsListDataConverter(BaseDataConverter):
    """Plannings list data converter"""

    def toWidgetValue(self, value):
        if not value:
            return ()
        return PlanningData.find_by_id(value).all()

    def toFieldValue(self, value):
        if not value:
            return []
        return value.split('|')


@widgettemplate_config(mode=INPUT_MODE,
                       template='templates/plannings-list-input.pt', layer=IFormLayer)
@widgettemplate_config(mode=DISPLAY_MODE,
                       template='templates/plannings-list-display.pt', layer=IFormLayer)
@implementer_only(IPlanningsListWidget)
class PlanningsListWidget(HTMLInputWidget, Widget):
    """Plannings list widget"""

    @property
    def values(self):
        return '|'.join((planning.token for planning in self.value or ()))

    @property
    def values_map(self):
        result = {}
        [result.update({value.token: value.title}) for value in self.value or ()]
        return json.dumps(result)


@adapter_config(context=(IPlanningsListField, IFormLayer), provides=IFieldWidget)
def PlanningsListFieldWidget(field, request):
    """Plannings list field widget factory"""
    return FieldWidget(field, PlanningsListWidget(request))


@widgettemplate_config(mode=INPUT_MODE,
                       template='templates/displays-input.pt', layer=IFormLayer)
@widgettemplate_config(mode=DISPLAY_MODE,
                       template='templates/displays-display.pt', layer=IFormLayer)
class PlanningTableDisplaysSelectWidget(HTMLFormElement, MultiWidget):
    """Planning table displays select widget"""

    @property
    def items(self):
        displays = self.field.bind(self.context).get(self.context) or {}
        vocabulary = getVocabularyRegistry().get(self.context, FOREST_OWNERS_CATEGORIES)
        for value, term in sorted(vocabulary.by_value.items(),
                                  key=lambda x: x[1].value):
            yield term, displays.get(value)

    def updateWidgets(self):
        pass

    def extract(self, default=NO_VALUE):
        if self.request.get('{}_marker'.format(self.name)) is None:
            return default
        result = []
        vocabulary = getVocabularyRegistry().get(self.context, FOREST_OWNERS_CATEGORIES)
        for value, term in vocabulary.by_value.items():
            result.append([
                (value,),
                ('true' if self.request.get('{}_{}'.format(self.name, value))
                 else 'false',)
            ])
        return result


def PlanningTableDisplaysSelectWidgetFactory(field, request):
    """Planning table displays select widget factory"""
    return FieldWidget(field, PlanningTableDisplaysSelectWidget(request))
