#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from pyramid.decorator import reify
from pyramid.httpexceptions import HTTPNotFound
from pyramid.url import resource_url
from pyramid.view import view_config
from z3c.form.field import Fields
from z3c.form.interfaces import DISPLAY_MODE, HIDDEN_MODE, INPUT_MODE
from z3c.table.interfaces import IColumn, IValues
from zope.interface import Interface, implementer
from zope.schema import Choice, TextLine

from onf_website.reference.forest.interfaces import FOREST_OWNERS_CATEGORIES
from onf_website.reference.planning import IPlanningTable
from onf_website.reference.planning.model.interfaces.planning import IInformationPlanning
from onf_website.reference.planning.model.planning import InformationPlanning, PARENT_SESSION as PLANNING_SESSION, \
    PlanningData
from onf_website.reference.planning.schema import PlanningsListField
from pyams_alchemy.engine import get_user_session
from pyams_content.interfaces import MANAGE_SITE_ROOT_PERMISSION
from pyams_content.reference.zmi.table import ReferenceTableHeaderAdapter
from pyams_form.form import ajax_config
from pyams_form.search import SearchForm, SearchResultsView, SearchView
from pyams_pagelet.interfaces import PageletCreatedEvent
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.interfaces import IContentSearch, IInnerPage, IPageHeader
from pyams_skin.interfaces.container import ITableElementEditor
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.skin import apply_skin
from pyams_skin.table import AttributeSwitcherColumn, NameColumn, \
    VisibilitySwitcherColumn
from pyams_utils.adapter import ContextRequestViewAdapter, adapter_config
from pyams_utils.interfaces import VIEW_SYSTEM_PERMISSION
from pyams_utils.url import absolute_url
from pyams_zmi.form import AdminDialogEditForm
from pyams_zmi.layer import IAdminLayer
from pyams_zmi.view import AdminView

__docformat__ = 'restructuredtext'

from onf_website import _


class IPlanningSearchFields(Interface):
    """Planning search fields"""

    label = TextLine(title=_("Free search"),
                     description=_("Search planning label for this text"),
                     required=False)

    planning_ids = PlanningsListField(title=_("Planning IDs"),
                                      description=_("List of searched plannings"),
                                      required=False)

    owners_category = Choice(title=_("Owner category"),
                             description=_("Owner category selection"),
                             required=False,
                             vocabulary=FOREST_OWNERS_CATEGORIES)


@implementer(IInnerPage)
class PlanningTableSearchForm(SearchForm):
    """Planning table search form"""

    legend = _("Planning search")

    fields = Fields(IPlanningSearchFields)
    sort_results = True

    prefix = 'search_form.'
    ajax_handler = 'planning-search-results.html'

    def __init__(self, context, request):
        super().__init__(context, request)
        request.registry.notify(PageletCreatedEvent(self))
        apply_skin(request, 'PyAMS admin skin')

    def updateWidgets(self, prefix=None):
        super().updateWidgets(prefix)


@adapter_config(context=(IPlanningTable, IPyAMSLayer, PlanningTableSearchForm),
                provides=IContentSearch)
class PlanningTableSearchFormResultsAdapter(ContextRequestViewAdapter):
    """Planning table search form results adapter"""

    def get_search_results(self, data):
        """Search results getter"""
        if not data:
            return
        session = get_user_session(PLANNING_SESSION)
        filters = []
        label = data.get('label')
        if label:
            filters.append(
                PlanningData.libelle_recherche.like('%{}%'.format(label.replace('-', ' ').upper())))
        planning_ids = data.get('planning_ids')
        if planning_ids:
            filters.append(PlanningData.id_nat_amgt.in_(planning_ids))
        category = data.get('owners_category')
        if category:
            filters.append(PlanningData.code_categorie_proprietaire == category)
        if not filters:
            return
        query = session.query(PlanningData, InformationPlanning) \
            .outerjoin(InformationPlanning,
                       PlanningData.id_nat_amgt == InformationPlanning.id_nat_amgt)
        yield from query.filter(*filters).limit(999)


@pagelet_config(name='contents.html',
                context=IPlanningTable, layer=IPyAMSLayer,
                permission=VIEW_SYSTEM_PERMISSION)
class PlanningTableSearchView(SearchView):
    """Planning table search view"""

    search_form_factory = PlanningTableSearchForm


@adapter_config(context=(IPlanningTable, IAdminLayer, Interface),
                provides=IPageHeader)
class PlanningTableHeaderAdapter(ReferenceTableHeaderAdapter):
    """Planning table header adapter"""


@view_config(name='planning-search-results.html',
             context=IPlanningTable, request_type=IPyAMSLayer,
             permission=VIEW_SYSTEM_PERMISSION)
class PlanningTableSearchResultsView(AdminView, SearchResultsView):
    """Planning table search results view"""

    title = _("Planning search results")
    search_form_factory = PlanningTableSearchForm

    sortOn = None
    startBatchingAt = 1000

    def __init__(self, context, request):
        super().__init__(context, request)
        request.registry.notify(PageletCreatedEvent(self))

    def get_element_id(self, element):
        return '{}::{}'.format(self.id, element[0].id_nat_amgt)

    @reify
    def data_attributes(self):
        attributes = super().data_attributes
        attributes.setdefault('table', {}).update({
            'data-ams-location': absolute_url(self.context, self.request),
            'data-ams-datatable-sorting': '1,asc',
            'data-ams-attribute-switcher': 'switch-planning-visibility.json'
        })
        attributes.setdefault('tr', {}).update({
            'data-ams-element-name': lambda x, col: x[0].id_nat_amgt
        })
        return attributes

    @reify
    def search_form(self):
        return self.search_form_factory(self.context, self.request)


@adapter_config(context=(IPlanningTable, IPyAMSLayer, PlanningTableSearchResultsView),
                provides=IValues)
class SearchResultsViewValuesAdapter(ContextRequestViewAdapter):
    """Search results view values adapter"""

    @property
    def values(self):
        return self.view.search_form.get_search_results() or ()


@adapter_config(name='visible',
                context=(IPlanningTable, IPyAMSLayer, PlanningTableSearchResultsView),
                provides=IColumn)
class PlanningVisibleColumn(VisibilitySwitcherColumn):
    """Planning visible column"""

    permission = MANAGE_SITE_ROOT_PERMISSION

    def get_icon(self, item):
        if isinstance(item, (list, tuple)):
            item = item[1]
        if item is None:
            return '<i class="{0}"></i>'.format(self.on_icon_class)
        return super().get_icon(item)

    def has_permission(self, item):
        return super().has_permission(self.table.context)

    def renderCell(self, item):
        permission = self.permission
        if permission and not self.request.has_permission(permission, context=self.table.context):
            return self.get_icon(item)
        return super(AttributeSwitcherColumn, self).renderCell(item[1])


@view_config(name='switch-planning-visibility.json',
             context=IPlanningTable, request_type=IPyAMSLayer,
             renderer='json', xhr=True,
             permission=MANAGE_SITE_ROOT_PERMISSION)
def switch_planning_visibility(request):
    """Switch planning visibility"""
    params = request.params
    id_nat_planning = params.get('object_name')
    if not id_nat_planning:
        raise HTTPNotFound()
    session = get_user_session(PLANNING_SESSION)
    planning, info = session.query(PlanningData, InformationPlanning) \
        .outerjoin(InformationPlanning, PlanningData.id_nat_amgt == InformationPlanning.id_nat_amgt) \
        .filter(PlanningData.id_nat_amgt == id_nat_planning) \
        .first()
    if planning is None:
        raise HTTPNotFound()
    if info is None:
        info = InformationPlanning(id_nat_amgt=id_nat_planning)
        info.visible = False
        session.add(info)
    else:
        info.visible = not info.visible
    return {
        'status': 'success',
        'visible': info.visible
    }


@adapter_config(name='name',
                context=(IPlanningTable, IPyAMSLayer, PlanningTableSearchResultsView),
                provides=IColumn)
class PlanningLabelColumn(NameColumn):
    """Planning label column"""

    _header = _("Usage label")
    attrName = 'nom_usage_amenagement'

    def getValue(self, obj):
        planning, info = obj
        return (info.libelle if info is not None else None) or planning.nom_usage_amenagement


@adapter_config(name='id_nat_amgt',
                context=(IPlanningTable, IPyAMSLayer, PlanningTableSearchResultsView),
                provides=IColumn)
class PlanningIDColumn(NameColumn):
    """Planning ID column"""

    _header = _("National ID")
    attrName = 'id_nat_amgt'

    weight = 30

    def getValue(self, obj):
        return super().getValue(obj[0])


@adapter_config(name='category',
                context=(IPlanningTable, IPyAMSLayer, PlanningTableSearchResultsView),
                provides=IColumn)
class PlanningCategoryColumn(NameColumn):
    """Planning owner category column"""

    _header = _("Owner category")
    attrName = 'categorie_proprietaire'

    weight = 40

    def getValue(self, obj):
        planning, info = obj
        return planning.categorie_proprietaire


@adapter_config(name='start',
                context=(IPlanningTable, IPyAMSLayer, PlanningTableSearchResultsView),
                provides=IColumn)
class PlanningStartColumn(NameColumn):
    """Planning start column"""

    _header = _("Start")
    attrName = 'start'

    weight = 50

    def getValue(self, obj):
        planning, info = obj
        return planning.annee_debut_amenagement


@adapter_config(name='end',
                context=(IPlanningTable, IPyAMSLayer, PlanningTableSearchResultsView),
                provides=IColumn)
class PlanningEndColumn(NameColumn):
    """Planning end column"""

    _header = _("End")
    attrName = 'end'

    weight = 51

    def getValue(self, obj):
        planning, info = obj
        return planning.annee_echeance_amenagement


@adapter_config(context=(Interface, IAdminLayer, PlanningTableSearchResultsView),
                provides=ITableElementEditor)
class PlanningElementEditor(ContextRequestViewAdapter):
    """Planning info element editor"""

    view_name = 'planning-properties.html'

    @property
    def url(self):
        planning = self.context[0]
        return resource_url(self.view.context, self.request, self.view_name,
                            query={'id_nat_amgt': planning.id_nat_amgt})

    modal_target = True


@pagelet_config(name='planning-properties.html',
                context=IPlanningTable, layer=IPyAMSLayer,
                permission=MANAGE_SITE_ROOT_PERMISSION)
@ajax_config(name='planning-properties.json',
             context=IPlanningTable, layer=IPyAMSLayer)
class PlanningPropertiesEditForm(AdminDialogEditForm):
    """Planning properties edit form"""

    prefix = 'planning_properties.'

    legend = _("Edit planning properties")

    fields = Fields(IInformationPlanning)
    ajax_handler = 'planning-properties.json'

    edit_permission = MANAGE_SITE_ROOT_PERMISSION

    @property
    def id_planning(self):
        id_planning = self.request.params.get('id_nat_amgt')
        if not id_planning:
            id_planning = self.request.params.get('{}widgets.id_nat_amgt'.format(self.prefix))
        return id_planning

    def check_mode(self):
        if not self.request.has_permission(MANAGE_SITE_ROOT_PERMISSION, context=self.context):
            return DISPLAY_MODE
        return INPUT_MODE

    def getContent(self):
        id_planning = self.id_planning
        if id_planning is None:
            raise HTTPNotFound()
        session = get_user_session(PLANNING_SESSION)
        planning, info = session.query(PlanningData, InformationPlanning) \
            .outerjoin(InformationPlanning, PlanningData.id_nat_amgt == InformationPlanning.id_nat_amgt) \
            .filter(PlanningData.id_nat_amgt == id_planning) \
            .first()
        if planning is None:
            raise HTTPNotFound()
        if info is None:
            info = InformationPlanning(id_nat_amgt=planning.id_nat_amgt,
                                    libelle=planning.nom_usage_amenagement)
            session.add(info)
        return info

    def updateWidgets(self, prefix=None):
        super().updateWidgets(prefix)
        id_planning = self.widgets.get('id_nat_amgt')
        if id_planning is not None:
            id_planning.mode = HIDDEN_MODE
