#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'


# import standard library

from pyramid.events import subscriber
from zope.component.interfaces import ISite
from zope.interface import implementer
from zope.lifecycleevent.interfaces import IObjectAddedEvent
from zope.schema.fieldproperty import FieldProperty
from zope.schema.vocabulary import SimpleTerm, SimpleVocabulary

from onf_website import _
# import interfaces
from onf_website.reference.location.interfaces.region import IRegion, IRegionTable
# import packages
from pyams_content.reference import ReferenceInfo, ReferenceTable
from pyams_i18n.interfaces import II18n
from pyams_sequence.reference import get_reference_target
from pyams_utils.registry import query_utility
from pyams_utils.request import check_request
from pyams_utils.traversing import get_parent
from pyams_utils.vocabulary import vocabulary_config


@implementer(IRegionTable)
class RegionTable(ReferenceTable):
    """Regions table"""


@subscriber(IObjectAddedEvent, context_selector=IRegionTable)
def handle_added_region_table(event):
    """Handle new region table"""
    site = get_parent(event.object, ISite)
    registry = site.getSiteManager()
    if registry is not None:
        registry.registerUtility(event.object, IRegionTable)


@implementer(IRegion)
class Region(ReferenceInfo):
    """Region record"""

    localization_title = FieldProperty(IRegion['localization_title'])
    appropriation_title = FieldProperty(IRegion['appropriation_title'])
    insee_code = FieldProperty(IRegion['insee_code'])
    logo = FieldProperty(IRegion['logo'])
    territory = FieldProperty(IRegion['territory'])
    departments = FieldProperty(IRegion['departments'])
    neighbors = FieldProperty(IRegion['neighbors'])

    def get_logo(self):
        if self.logo is not None:
            return get_reference_target(self.logo)


@vocabulary_config(name='ONF regions')
class RegionVocabulary(SimpleVocabulary):
    """Regions vocabulary"""

    def __init__(self, context=None):
        terms = []
        table = query_utility(IRegionTable)
        if table is not None:
            request = check_request()
            terms = [SimpleTerm(v,
                                token=v.__name__,
                                title=II18n(v).query_attribute('title', request=request))
                     for v in table.values()]
        super(RegionVocabulary, self).__init__(terms)


@vocabulary_config(name='ONF regions with departments')
class FullRegionsVocabulary(SimpleVocabulary):
    """Regions with departments"""

    def __init__(self, context=None):
        terms = []
        table = query_utility(IRegionTable)
        if table is not None:
            request = check_request()
            translate = request.localizer.translate
            for region in sorted(table.values(), key=lambda x: II18n(x).query_attribute('title', request=request)):
                title = II18n(region).query_attribute('appropriation_title', request=request) or \
                        II18n(region).query_attribute('title', request=request)
                terms.append(SimpleTerm(region,
                                        token='region::{0}'.format(region.__name__),
                                        title=translate(_("{0}{1} (all departments)")).format(title[0].upper(),
                                                                                              title[1:])))
                if region.insee_code:
                    for dept in region.departments or ():
                        terms.append(SimpleTerm(dept,
                                                token='dept::{}::{}'.format(region.__name__, dept.__name__),
                                                title=' - {0} ({1})'.format(
                                                    II18n(dept).query_attribute('title', request=request),
                                                    dept.insee_code)))
        super(FullRegionsVocabulary, self).__init__(terms)
