#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'


# import standard library

# import interfaces
from onf_website.reference.location.interfaces.department import IDepartmentTable, IDepartment
from pyams_i18n.interfaces import II18n
from zope.component.interfaces import ISite
from zope.lifecycleevent.interfaces import IObjectAddedEvent

# import packages
from pyams_content.reference import ReferenceTable, ReferenceInfo
from pyams_sequence.reference import get_reference_target
from pyams_utils.registry import query_utility
from pyams_utils.request import check_request
from pyams_utils.traversing import get_parent
from pyams_utils.vocabulary import vocabulary_config
from pyramid.events import subscriber
from zope.interface import implementer
from zope.schema.fieldproperty import FieldProperty
from zope.schema.vocabulary import SimpleVocabulary, SimpleTerm


@implementer(IDepartmentTable)
class DepartmentTable(ReferenceTable):
    """Departments table"""


@subscriber(IObjectAddedEvent, context_selector=IDepartmentTable)
def handle_added_department_table(event):
    """Handle new department table"""
    site = get_parent(event.object, ISite)
    registry = site.getSiteManager()
    if registry is not None:
        registry.registerUtility(event.object, IDepartmentTable)


@implementer(IDepartment)
class Department(ReferenceInfo):
    """Department record"""

    localization_title = FieldProperty(IDepartment['localization_title'])
    appropriation_title = FieldProperty(IDepartment['appropriation_title'])
    insee_code = FieldProperty(IDepartment['insee_code'])
    logo = FieldProperty(IDepartment['logo'])
    region = FieldProperty(IDepartment['region'])
    neighbors = FieldProperty(IDepartment['neighbors'])

    def get_logo(self):
        if self.logo is not None:
            return get_reference_target(self.logo)


@vocabulary_config(name='ONF departments')
class DepartmentVocabulary(SimpleVocabulary):
    """Departments vocabulary"""

    def __init__(self, context=None):
        table = query_utility(IDepartmentTable)
        if table is not None:
            request = check_request()
            terms = [SimpleTerm(v,
                                token=v.__name__,
                                title='{0} ({1})'.format(II18n(v).query_attribute('title', request=request),
                                                         v.insee_code))
                     for v in table.values()]
        else:
            terms = []
        super(DepartmentVocabulary, self).__init__(terms)
