#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'


# import standard library

# import interfaces
from onf_website.reference.location.interfaces.country import ICountryTable, ICountry
from pyams_i18n.interfaces import II18n
from zope.component.interfaces import ISite
from zope.lifecycleevent.interfaces import IObjectAddedEvent

# import packages
from pyams_content.reference import ReferenceTable, ReferenceInfo
from pyams_utils.registry import query_utility
from pyams_utils.request import check_request
from pyams_utils.traversing import get_parent
from pyams_utils.vocabulary import vocabulary_config
from pyramid.events import subscriber
from zope.interface import implementer
from zope.schema.fieldproperty import FieldProperty
from zope.schema.vocabulary import SimpleVocabulary, SimpleTerm


@implementer(ICountryTable)
class CountryTable(ReferenceTable):
    """Countries table"""


@subscriber(IObjectAddedEvent, context_selector=ICountryTable)
def handle_added_country_table(event):
    """Handle new country table"""
    site = get_parent(event.object, ISite)
    registry = site.getSiteManager()
    if registry is not None:
        registry.registerUtility(event.object, ICountryTable)


@implementer(ICountry)
class Country(ReferenceInfo):
    """Country record"""

    localization_title = FieldProperty(ICountry['localization_title'])
    appropriation_title = FieldProperty(ICountry['appropriation_title'])
    insee_code = FieldProperty(ICountry['insee_code'])


@vocabulary_config(name='ONF countries')
class CountryVocabulary(SimpleVocabulary):
    """Countries vocabulary"""

    def __init__(self, context=None):
        table = query_utility(ICountryTable)
        if table is not None:
            request = check_request()
            terms = [SimpleTerm(v,
                                token=v.__name__,
                                title=II18n(v).query_attribute('title', request=request))
                     for v in table.values()]
        else:
            terms = []
        super(CountryVocabulary, self).__init__(terms)
