#
# Copyright (c) 2015-2022 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS_*** module

"""

__docformat__ = 'restructuredtext'

from elasticsearch_dsl import Q, Search
from pyramid.view import view_config
from pyramid_es import get_client

from onf_website.reference.forest import IForestTable
from pyams_content.root import ISiteRoot
from pyams_default_theme.features.sitemap import ISitemapExtension
from pyams_skin.layer import IPyAMSUserLayer
from pyams_utils.adapter import ContextRequestAdapter, adapter_config
from pyams_utils.registry import query_utility
from pyams_workflow.interfaces import IWorkflow


@adapter_config(name='forests',
                context=(ISiteRoot, IPyAMSUserLayer),
                provides=ISitemapExtension)
class SiteRootForestsSitemap(ContextRequestAdapter):
    """Site root forests sitemap extension"""

    @property
    def source(self):
        return query_utility(IForestTable)


@view_config(name='sitemap.xml',
             context=IForestTable, request_type=IPyAMSUserLayer,
             renderer='templates/forests-sitemap.pt')
class ForestTableSitemapView:
    """Forest table sitemap view"""

    def __init__(self, request):
        self.request = request

    def get_contents(self):
        """Get indexed contents with location forest"""
        wf_params = []
        for workflow in self.request.registry.getAllUtilitiesRegisteredFor(IWorkflow):
            wf_params.extend(workflow.visible_states)
        params = \
            Q('terms',
              **{'workflow.status': wf_params}) & \
            Q('range',
              **{'workflow.effective_date': {'lte': 'now/m'}}) & \
            (Q('bool',
               must=Q('range',
                      **{'workflow.push_end_date': {'gte': 'now/m'}})) |
             Q('bool',
               must_not=Q('exists',
                          **{'field': 'workflow.push_end_date'}))) & \
            Q('bool',
              must=Q('exists',
                     **{'field': 'location.forests'}))
        es_client = get_client(self.request)
        contents = {}
        for content in Search(using=es_client.es, index=es_client.index) \
                .params(request_timeout=60) \
                .query(params) \
                .source([
                    'location.forests',
                    'workflow.modified_date'
                ]):
            for forest in content.location.forests or ():
                if forest in contents:
                    contents[forest] = max(contents[forest], content.workflow.modified_date)
                else:
                    contents[forest] = content.workflow.modified_date
        return contents

    def __call__(self):
        self.request.response.content_type = 'text/xml'
        self.contents = self.get_contents()
        return {}
