#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'


# import standard library
import string
from datetime import date

# import interfaces
from onf_website.reference.agent.model.interfaces import IAgent
from zope.schema.interfaces import ITitledTokenizedTerm

# import packages
from onf_website.reference.orga.model import Structure
from pyams_alchemy import Base
from pyams_alchemy.engine import get_user_session
from pyams_alchemy.mixin import DynamicSchemaMixin
from sqlalchemy import Column, ForeignKey
from sqlalchemy import Integer, Unicode, Date
from sqlalchemy.orm import relation
from sqlalchemy.sql import and_, or_
from zope.interface import implementer


PARENT_SCHEMA = 'nemovita'
PARENT_SESSION = 'NEMOVITA'


@implementer(IAgent, ITitledTokenizedTerm)
class Agent(DynamicSchemaMixin, Base):
    """Base ONF agent class"""

    __tablename__ = 'vm_bu_agent'
    __schema__ = PARENT_SCHEMA

    matricule_bu = Column('matricule_bu', Integer, primary_key=True)
    matricule_perso = Column('matricule_perso', Unicode(5))
    ldap_username = Column('ldap_username', Unicode(32))
    date_entree = Column('d_entree_onf', Date)
    date_sortie = Column('d_sortie_onf', Date)
    code_civilite = Column('civilite', Unicode(3))
    nom = Column('nom', Unicode(80))
    prenom = Column('prenom', Unicode(40))
    code_structure = Column('code_structure_affectation', Unicode(6))
    statut = Column('statut', Unicode(3))
    code_poste = Column('code_poste', Unicode(8))
    complement_adresse = Column('adrpro_comp_adresse', Unicode(38))
    num_voie = Column('adrpro_n_voie', Integer)
    ext_voie = Column('adrpro_ext_n_voie', Unicode(1))
    nature_voie = Column('adrpro_nature_nom_voie', Unicode(32))
    nom_commune = Column('adrpro_nom_commune', Unicode(32))
    code_postal = Column('adrpro_code_postal', Unicode(10))
    bureau_distrib = Column('adrpro_bureau_distributeur', Unicode(32))
    pays = Column('adrpro_nom_pays', Unicode(38))
    telephone = Column('tel_pro', Unicode(20))
    fax = Column('fax_pro', Unicode(20))
    mobile = Column('mobile_pro', Unicode(20))
    mail = Column('mail_pro', Unicode(100))
    #matricule_manager = Column('matricule_bu_manager', Integer)
    code_metier = Column('code_metier', Unicode(50))

    @property
    def value(self):
        return self.matricule_bu

    @property
    def token(self):
        return str(self.matricule_bu)

    @property
    def shortname(self):
        return u'%s %s' % (self.nom, '-'.join(map(string.capitalize, self.prenom.split('-'))))

    @property
    def title(self):
        return self.shortname

    @property
    def longname(self):
        return self.shortname

    @classmethod
    def get(cls, matricule, session=PARENT_SESSION):
        return Agent.find_by_matricule(matricule, session)

    @classmethod
    def find(cls, query, session=PARENT_SESSION):
        if isinstance(query, dict):
            matricule = query.get('matricule')
            name = query.get('name', '').strip().upper()
            ldap_username = query.get('ldap_username').strip()
        else:
            try:
                matricule = int(query.strip().upper())
            except:
                matricule = None
            name = query.strip().upper()
            ldap_username = None
        if ' ' in name:
            name = [ n + '%' for n in name.split() ]
        params = []
        if matricule:
            params.append(Agent.matricule_bu == matricule)
        if name:
            if isinstance(name, (list, tuple)):
                params.append(or_(and_(Agent.nom.like(name[0]),
                                       Agent.prenom.like(' '.join(name[1:]))),
                                  and_(Agent.prenom.like(name[0]),
                                       Agent.nom.like(' '.join(name[1:])))))
            else:
                params.append(or_(Agent.nom.like(name + '%'),
                                  Agent.prenom.like(name + '%')))
        if ldap_username:
            params.append(Agent.ldap_username == ldap_username)
        params = [or_(*params), ]
        session = get_user_session(session)
        return session.query(Agent).filter(and_(*params))

    @classmethod
    def find_by_matricule(cls, matricule, session=PARENT_SESSION):
        if isinstance(matricule, (list, tuple, set)):
            params = Agent.matricule_bu.in_(matricule)
        else:
            params = Agent.matricule_bu == matricule
        session = get_user_session(session)
        return session.query(Agent).filter(params)

    @classmethod
    def find_by_name(cls, name, session=PARENT_SESSION):
        name = name.strip().upper()
        session = get_user_session(session)
        return session.query(Agent).filter(or_(Agent.nom.like(name + '%'),
                                               Agent.prenom.like(name + '%')))

    @classmethod
    def from_structure(cls, structure, recurse=True, session=PARENT_SESSION):
        """Get all agents from given structure"""
        if isinstance(structure, str):
            structure = Structure.get(structure, session=session).first()
        if structure is None:
            return []
        if recurse:
            structures = structure.getChilds(session=session)
        else:
            structures = [structure]
        session = get_user_session(session)
        return session.query(Agent).filter(Agent.code_structure.in_([structure.code for structure in structures]))

    def get_structure(self, reference_date=None, session=PARENT_SESSION):
        if reference_date is None:
            reference_date = date.today()
        session = get_user_session(session)
        return Structure.get(self.code_structure, reference_date, session).first()


Agent.matricule_manager = Column('matricule_bu_manager', Integer, ForeignKey(Agent.matricule_bu))
Agent.manager = relation(Agent, primaryjoin=Agent.matricule_manager == Agent.matricule_bu)

Agent.structure = relation(Structure,
                           foreign_keys=Agent.code_structure,
                           primaryjoin=and_(Agent.code_structure == Structure.code,
                                            and_(Structure.date_debut <= date.today(),
                                                 or_(Structure.date_fin == None,
                                                     Structure.date_fin >= date.today()))))
