#
# Copyright (c) 2015-2023 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS_*** module

"""

import json

from pyramid.decorator import reify
from pyramid.httpexceptions import HTTPForbidden
from pyramid.view import view_config
from z3c.form import field
from z3c.form.browser.orderedselect import OrderedSelectFieldWidget
from z3c.form.field import Fields
from z3c.table.interfaces import IColumn, IValues
from zope.interface import alsoProvides
from zope.interface import implementer

from onf_website.features.search.skin.interfaces.filter import ICollectionFilter, IContentTypeFilter, IFilter, \
    IFilterContainer, IFilterContainerTarget, ILocationFilter, \
    ISearchResultsPortletAdvancedFiltersRendererSettings, ITagFilter, ITargetFilter, IThemeFilter
from pyams_content.zmi import pyams_content
from pyams_form.form import AJAXAddForm, ajax_config
from pyams_form.interfaces.form import IFormContextPermissionChecker, IInnerSubForm
from pyams_form.security import ProtectedFormObjectMixin
from pyams_i18n.column import I18nAttrColumn
from pyams_pagelet.pagelet import pagelet_config
from pyams_portal.interfaces import MANAGE_TEMPLATE_PERMISSION
from pyams_portal.zmi.portlet import PortletRendererPropertiesEditForm
from pyams_skin.container import delete_container_element, switch_element_visibility
from pyams_skin.event import get_json_table_refresh_event, get_json_table_row_refresh_event
from pyams_skin.interfaces.viewlet import IToolbarAddingMenu
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.table import BaseTable, I18nColumn, SorterColumn, TrashColumn, VisibilitySwitcherColumn
from pyams_skin.viewlet.toolbar import ToolbarMenuItem
from pyams_utils.adapter import ContextRequestViewAdapter, adapter_config
from pyams_utils.factory import create_object
from pyams_utils.fanstatic import get_resource_path
from pyams_utils.interfaces.data import IObjectData
from pyams_utils.traversing import get_parent
from pyams_utils.url import absolute_url
from pyams_viewlet.viewlet import viewlet_config
from pyams_zmi.form import AdminDialogAddForm, AdminDialogEditForm
from pyams_zmi.interfaces import IPropertiesEditForm
from pyams_zmi.layer import IAdminLayer
from pyams_zmi.zmi.table import InnerTableView

from onf_website import _


class SearchResultsPortletAdvancedFiltersRendererSettingsFiltersTable(ProtectedFormObjectMixin, BaseTable):
    """Search results portlet advanced filters renderer settings filters table"""

    prefix = 'filters'

    hide_header = True
    hide_body_toolbar = True
    sortOn = None

    @property
    def cssClasses(self):
        classes = ['table', 'table-bordered', 'table-striped', 'table-hover', 'table-tight']
        permission = self.permission
        if (not permission) or self.request.has_permission(permission, context=self.context):
            classes.append('table-dnd')
        return {'table': ' '.join(classes)}
 
    @reify
    def data_attributes(self):
        attributes = super().data_attributes
        attributes.setdefault('table', {}).update({
            'data-ams-location': absolute_url(self.context, self.request, "++filter++"),
            'data-ams-tablednd-drag-handle': 'td.sorter',
            'data-ams-tablednd-drop-target': 'set-filters-order.json'
        })
        attributes.setdefault('td', {}).update({
            'data-ams-attribute-switcher': self.get_switcher_target
        })
        return attributes

    @staticmethod
    def get_switcher_target(element, column):
        if column.__name__ == 'show-hide':
            return 'switch-filter-visibility.json'


@adapter_config(required=(ISearchResultsPortletAdvancedFiltersRendererSettings, IAdminLayer,
                          SearchResultsPortletAdvancedFiltersRendererSettingsFiltersTable),
                provides=IValues)
class SearchResultsPortletAdvancedFiltersRendererSettingsFiltersTableValues(ContextRequestViewAdapter):
    """Search results portlet advanced filters renderer settings filters table values adapter"""

    @property
    def values(self):
     
        return IFilterContainer(self.context).values()


@adapter_config(name='show-hide',
                required=(ISearchResultsPortletAdvancedFiltersRendererSettings, IAdminLayer,
                          SearchResultsPortletAdvancedFiltersRendererSettingsFiltersTable),
                provides=IColumn)
class SearchResultsPortletAdvancedFiltersRendererSettingsFiltersVisibleColumn(
        ProtectedFormObjectMixin, VisibilitySwitcherColumn):
    """Search results portlet advanced filters renderer settings filters visibility column"""


@adapter_config(name='sorter',
                required=(ISearchResultsPortletAdvancedFiltersRendererSettings, IAdminLayer,
                          SearchResultsPortletAdvancedFiltersRendererSettingsFiltersTable),
                provides=IColumn)
class MenusTableSorterColumn(ProtectedFormObjectMixin, SorterColumn):
    """Menus table sorter column"""


@adapter_config(name='trash',
                required=(IFilterContainerTarget, IPyAMSLayer,
                          SearchResultsPortletAdvancedFiltersRendererSettingsFiltersTable),
                provides=IColumn)
class MenusTableTrashColumn(ProtectedFormObjectMixin, TrashColumn):
    """Menus table trash column"""


@view_config(name='delete-element.json',
             context=IFilterContainer, request_type=IPyAMSLayer,
             renderer='json', xhr=True)
def delete_menu(request):
    """Delete menu"""
    permission = IFormContextPermissionChecker(request.context).edit_permission
    if not request.has_permission(permission):
        raise HTTPForbidden()
    return delete_container_element(request, ignore_permission=True)


@view_config(name='set-filters-order.json',
             context=IFilterContainer, request_type=IPyAMSLayer,
             renderer='json', xhr=True)
def set_menus_order(request):
    """Update menus order"""
    permission = IFormContextPermissionChecker(request.context).edit_permission
    if not request.has_permission(permission):
        raise HTTPForbidden()
    order = list(map(str, json.loads(request.params.get('names'))))
    request.context.updateOrder(order)
    return {'status': 'success'}


@view_config(name='switch-filter-visibility.json',
             context=IFilterContainer, request_type=IPyAMSLayer,
             renderer='json', xhr=True)
def set_filter_visibility(request):
    """Set filter visibility"""
    permission = IFormContextPermissionChecker(request.context).edit_permission
    if not request.has_permission(permission):
        raise HTTPForbidden()
    return switch_element_visibility(request, IFilterContainer)


@adapter_config(name='name',
                required=(ISearchResultsPortletAdvancedFiltersRendererSettings, IAdminLayer,
                          SearchResultsPortletAdvancedFiltersRendererSettingsFiltersTable),
                provides=IColumn)
class SearchResultsPortletAdvancedFiltersRendererSettingsFiltersNameColumn(I18nColumn, I18nAttrColumn):
    """Search results portlet advanced filters renderer settings filters table name column"""

    _header = _("Label")
    attrName = 'label'

    weight = 10


@adapter_config(name='search-filters',
                required=(ISearchResultsPortletAdvancedFiltersRendererSettings, IAdminLayer,
                          PortletRendererPropertiesEditForm),
                provides=IInnerSubForm)
class SearchResultsPortletAdvancedFiltersRendererSettingsFiltersView(InnerTableView):
    """Search results portlet advanced filters renderer settings filters view"""

    title = _("Search filters")

    table_class = SearchResultsPortletAdvancedFiltersRendererSettingsFiltersTable
    weight = 10


#
# Custom filters
#

class BaseFilterAddForm(AdminDialogAddForm):
    """Base add form for all filter types."""

    legend = _("Add new filter")

    filter_interface = IFilter

    @property
    def fields(self):
        return Fields(self.filter_interface).omit('visible')

    def create(self, data):
        return create_object(self.filter_interface)

    def add(self, obj):
        IFilterContainer(self.context).add(obj)

    def get_ajax_output(self, changes):
        if changes is None:
            return None
        return {
            'status': 'success',
            'message': self.request.localizer.translate(_("Filter was correctly added.")),
            'events': [
                get_json_table_refresh_event(self.context, self.request,
                                             SearchResultsPortletAdvancedFiltersRendererSettingsFiltersTable)
            ]
        }


class BaseThesaurusFilterAddForm(BaseFilterAddForm):
    """Base thesaurus filter add form"""

    def updateWidgets(self, prefix=None):
        if self.request.method == 'POST':
            param_name = '{}widgets.thesaurus_name:list'.format(self.prefix)
            param_value = self.request.params.get(param_name)
            if param_value is not None:
                self.request.headers['X-Thesaurus-Name'] = param_value
        super().updateWidgets(prefix)
        widget = self.widgets.get('thesaurus_name')
        if widget is not None:
            widget.prompt = True
            widget.promptMessage = _("Please select a thesaurus...")
            widget.object_data = {
                'ams-plugins': 'pyams_content',
                'ams-plugin-pyams_content-src': get_resource_path(pyams_content),
                'ams-change-handler': 'PyAMS_content.themes.updateThesaurus'
            }
            alsoProvides(widget, IObjectData)


class BaseFilterEditForm(AdminDialogEditForm):
    """Base edit form for all facet types."""

    legend = _("Edit facet properties")

    filter_interface = IFilter

    @property
    def fields(self):
        return Fields(self.filter_interface).omit('visible')
    
    edit_permission = MANAGE_TEMPLATE_PERMISSION

    def get_ajax_output(self, changes):
        output = super(BaseFilterEditForm, self).get_ajax_output(changes)
        if changes:
            settings = get_parent(self.context, ISearchResultsPortletAdvancedFiltersRendererSettings)
            output.setdefault('events', []).append(
                get_json_table_row_refresh_event(settings, self.request,
                                                 SearchResultsPortletAdvancedFiltersRendererSettingsFiltersTable,
                                                 self.context))
        return output


class BaseThesaurusFilterEditForm(BaseFilterEditForm):
    """Base thesaurus filter edit form"""

    def updateWidgets(self, prefix=None):
        if self.request.method == 'POST':
            param_name = '{}widgets.thesaurus_name:list'.format(self.prefix)
            param_value = self.request.params.get(param_name)
            if param_value is not None:
                self.request.headers['X-Thesaurus-Name'] = param_value
        else:
            param_value = self.filter_interface(self.context).thesaurus_name
            if param_value:
                self.request.headers['X-Thesaurus-Name'] = param_value
        super().updateWidgets(prefix)
        widget = self.widgets['thesaurus_name']
        widget.object_data = {
            'ams-plugins': 'pyams_content',
            'ams-plugin-pyams_content-src': get_resource_path(pyams_content),
            'ams-change-handler': 'PyAMS_content.themes.updateThesaurus'
        }
        alsoProvides(widget, IObjectData)


#
# Content-type filters
#

@viewlet_config(name='add-content-type-filter.menu',
                context=IFilterContainerTarget, layer=IAdminLayer,
                view=SearchResultsPortletAdvancedFiltersRendererSettingsFiltersView,
                manager=IToolbarAddingMenu, weight=20,
                permission=MANAGE_TEMPLATE_PERMISSION)
class ContentTypeFilterAddMenu(ToolbarMenuItem):
    """Content-type filter add menu"""

    label = _("Add content-type filter")
    label_css_class = ''

    url = 'add-content-type-filter.html'
    modal_target = True


@pagelet_config(name='add-content-type-filter.html',
                context=IFilterContainerTarget, layer=IPyAMSLayer,
                permission=MANAGE_TEMPLATE_PERMISSION)
@ajax_config(name='add-content-type-filter.json',
             context=IFilterContainerTarget, layer=IPyAMSLayer,
             base=AJAXAddForm)
class ContentTypeFilterAddForm(BaseFilterAddForm):
    """Content-type filter add form"""
    
    legend = _("Add new content-type filter")
    filter_interface = IContentTypeFilter


@pagelet_config(name='properties.html',
                context=IContentTypeFilter, layer=IPyAMSLayer,
                permission=MANAGE_TEMPLATE_PERMISSION)
@ajax_config(name='properties.json',
             context=IContentTypeFilter, layer=IPyAMSLayer)
@implementer(IPropertiesEditForm)
class ContentTypeFilterPropertiesEditForm(BaseFilterEditForm):
    """Content-type filter properties edit form"""

    legend = _("Edit filter properties")
    filter_interface = IContentTypeFilter


#
# Tags filters
#

@viewlet_config(name='add-tags-filter.menu',
                context=IFilterContainerTarget, layer=IAdminLayer,
                view=SearchResultsPortletAdvancedFiltersRendererSettingsFiltersView,
                manager=IToolbarAddingMenu, weight=30,
                permission=MANAGE_TEMPLATE_PERMISSION)
class TagFilterAddMenu(ToolbarMenuItem):
    """Content-type filter add menu"""

    label = _("Add tags filter")
    label_css_class = ''

    url = 'add-tags-filter.html'
    modal_target = True


@pagelet_config(name='add-tags-filter.html',
                context=IFilterContainerTarget, layer=IPyAMSLayer,
                permission=MANAGE_TEMPLATE_PERMISSION)
@ajax_config(name='add-tags-filter.json',
             context=IFilterContainerTarget, layer=IPyAMSLayer,
             base=AJAXAddForm)
class TagFilterAddForm(BaseThesaurusFilterAddForm):
    """Content-type filter add form"""

    legend = _("Add new tag filter")

    filter_interface = ITagFilter


@pagelet_config(name='properties.html',
                context=ITagFilter, layer=IPyAMSLayer,
                permission=MANAGE_TEMPLATE_PERMISSION)
@ajax_config(name='properties.json',
             context=ITagFilter, layer=IPyAMSLayer)
@implementer(IPropertiesEditForm)
class TagFilterPropertiesEditForm(BaseThesaurusFilterEditForm):
    """Content-type filter properties edit form"""

    legend = _("Edit filter properties")

    filter_interface = ITagFilter


#
# Collections filters
#

@viewlet_config(name='add-collection-filter.menu',
                context=IFilterContainerTarget, layer=IAdminLayer,
                view=SearchResultsPortletAdvancedFiltersRendererSettingsFiltersView,
                manager=IToolbarAddingMenu, weight=40,
                permission=MANAGE_TEMPLATE_PERMISSION)
class CollectionFilterAddMenu(ToolbarMenuItem):
    """Content-type filter add menu"""

    label = _("Add collection filter")
    label_css_class = ''

    url = 'add-collection-filter.html'
    modal_target = True


@pagelet_config(name='add-collection-filter.html',
                context=IFilterContainerTarget, layer=IPyAMSLayer,
                permission=MANAGE_TEMPLATE_PERMISSION)
@ajax_config(name='add-collection-filter.json',
             context=IFilterContainerTarget, layer=IPyAMSLayer,
             base=AJAXAddForm)
class CollectionFilterAddForm(BaseThesaurusFilterAddForm):

    legend = _("Add new collection filter")

    filter_interface = ICollectionFilter


@pagelet_config(name='properties.html',
                context=ICollectionFilter, layer=IPyAMSLayer,
                permission=MANAGE_TEMPLATE_PERMISSION)
@ajax_config(name='properties.json',
             context=ICollectionFilter, layer=IPyAMSLayer)
@implementer(IPropertiesEditForm)
class CollectionFilterEditForm(BaseThesaurusFilterEditForm):

    legend = _("Edit collection filter properties")

    filter_interface = ICollectionFilter


#
# Themes filters
#

@viewlet_config(name='add-theme-filter.menu',
                context=IFilterContainerTarget, layer=IAdminLayer,
                view=SearchResultsPortletAdvancedFiltersRendererSettingsFiltersView,
                manager=IToolbarAddingMenu, weight=50,
                permission=MANAGE_TEMPLATE_PERMISSION)
class ThemeFilterAddMenu(ToolbarMenuItem):
    """Content-type filter add menu"""

    label = _("Add theme filter")
    label_css_class = ''

    url = 'add-theme-filter.html'
    modal_target = True


@pagelet_config(name='add-theme-filter.html',
                context=IFilterContainerTarget, layer=IPyAMSLayer,
                permission=MANAGE_TEMPLATE_PERMISSION)
@ajax_config(name='add-theme-filter.json',
             context=IFilterContainerTarget, layer=IPyAMSLayer,
             base=AJAXAddForm)
class ThemeFilterAddForm(BaseThesaurusFilterAddForm):

    legend = _("Add new theme filter")

    filter_interface = IThemeFilter


@pagelet_config(name='properties.html',
                context=IThemeFilter, layer=IPyAMSLayer,
                permission=MANAGE_TEMPLATE_PERMISSION)
@ajax_config(name='properties.json',
             context=IThemeFilter, layer=IPyAMSLayer)
@implementer(IPropertiesEditForm)
class ThemeFilterEditForm(BaseThesaurusFilterEditForm):

    legend = _("Edit theme filter properties")

    filter_interface = IThemeFilter


#
# Hearing filters
#

@viewlet_config(name='add-hearing-filter.menu',
                context=IFilterContainerTarget, layer=IAdminLayer,
                view=SearchResultsPortletAdvancedFiltersRendererSettingsFiltersView,
                manager=IToolbarAddingMenu, weight=60,
                permission=MANAGE_TEMPLATE_PERMISSION)
class HearingFilterAddMenu(ToolbarMenuItem):
    """hearing filter add menu"""

    label = _("Add hearing filter")
    label_css_class = ''

    url = 'add-hearing-filter.html'
    modal_target = True


@pagelet_config(name='add-hearing-filter.html',
                context=IFilterContainerTarget, layer=IPyAMSLayer,
                permission=MANAGE_TEMPLATE_PERMISSION)
@ajax_config(name='add-hearing-filter.json',
             context=IFilterContainerTarget, layer=IPyAMSLayer,
             base=AJAXAddForm)
class HearingFilterAddForm(BaseFilterAddForm):

    legend = _("Add new hearing filter")

    filter_interface = ITargetFilter

    @property
    def fields(self):
        fields = super().fields
        fields['targets'].widgetFactory = OrderedSelectFieldWidget
        return fields

    def updateWidgets(self, prefix=None):
        super().updateWidgets(prefix)
        if 'targets' in self.widgets:
            self.widgets['targets'].label = _("Targets")


@pagelet_config(name='properties.html',
                context=ITargetFilter, layer=IPyAMSLayer,
                permission=MANAGE_TEMPLATE_PERMISSION)
@ajax_config(name='properties.json',
             context=ITargetFilter, layer=IPyAMSLayer)
@implementer(IPropertiesEditForm)
class HearingFilterEditForm(BaseFilterEditForm):

    legend = _("Edit hearing filter properties test")

    filter_interface = ITargetFilter

    @property
    def fields(self):
        fields = super().fields
        fields['targets'].widgetFactory = OrderedSelectFieldWidget
        return fields

    def updateWidgets(self, prefix=None):
        super().updateWidgets(prefix)
        if 'targets' in self.widgets:
            self.widgets['targets'].label = _("Targets")


#
# Location filters
#

@viewlet_config(name='add-location-filter.menu',
                context=IFilterContainerTarget, layer=IAdminLayer,
                view=SearchResultsPortletAdvancedFiltersRendererSettingsFiltersView,
                manager=IToolbarAddingMenu, weight=70,
                permission=MANAGE_TEMPLATE_PERMISSION)
class LocationFilterAddMenu(ToolbarMenuItem):
    """Content-type filter add menu"""

    label = _("Add location filter")
    label_css_class = ''

    url = 'add-location-filter.html'
    modal_target = True


@pagelet_config(name='add-location-filter.html',
                context=IFilterContainerTarget, layer=IPyAMSLayer,
                permission=MANAGE_TEMPLATE_PERMISSION)
@ajax_config(name='add-location-filter.json',
             context=IFilterContainerTarget, layer=IPyAMSLayer,
             base=AJAXAddForm)
class LocationFilterAddForm(BaseFilterAddForm):

    legend = _("Add new location filter")

    filter_interface = ILocationFilter


@pagelet_config(name='properties.html',
                context=ILocationFilter, layer=IPyAMSLayer,
                permission=MANAGE_TEMPLATE_PERMISSION)
@ajax_config(name='properties.json',
             context=ILocationFilter, layer=IPyAMSLayer)
@implementer(IPropertiesEditForm)
class Location(BaseFilterEditForm):

    legend = _("Edit location filter properties")

    filter_interface = ILocationFilter
