#
# Copyright (c) 2015-2021 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS_*** module

"""

__docformat__ = 'restructuredtext'

import json
from z3c.form.browser.checkbox import SingleCheckBoxFieldWidget
from z3c.form.field import Fields
from zope.interface import alsoProvides
from zope.schema import getFieldsInOrder

from onf_website.features.search.skin.interfaces import \
    ICardsWithHeadFiltersSearchResultsPortletRendererSettings, \
    IHeadFiltersSearchResultsPortletRendererSettings, \
    IMapSearchResultsPortletRendererSettings, \
    IMapWithHeadFiltersSearchResultsPortletRendererSettings
from pyams_content.component.theme.interfaces import IThemesManager, IThemesManagerTarget
from pyams_content.shared.common.types import get_all_data_types
from pyams_form.group import NamedWidgetsGroup
from pyams_form.interfaces.form import IFormManager
from pyams_form.manager import DialogEditFormFormManager
from pyams_form.widget import HiddenSelect2FieldWidget
from pyams_gis.interfaces.configuration import IMapConfiguration
from pyams_portal.zmi.portlet import PortletRendererPropertiesEditForm
from pyams_skin.layer import IPyAMSLayer
from pyams_utils.adapter import adapter_config
from pyams_utils.interfaces.data import IObjectData
from pyams_utils.traversing import get_parent

from onf_website import _


@adapter_config(context=(IHeadFiltersSearchResultsPortletRendererSettings, IPyAMSLayer,
                         PortletRendererPropertiesEditForm),
                provides=IFormManager)
class HeadFiltersSearchResultsPortletRendererSettingsFormManager(DialogEditFormFormManager):
    """Head filters search results portlet renderer settings form manager"""

    def getFields(self):
        fields = Fields(IHeadFiltersSearchResultsPortletRendererSettings)
        fields['ignored_data_types'].widgetFactory = HiddenSelect2FieldWidget
        return fields

    def updateWidgets(self, prefix=None):
        super().updateWidgets(prefix)
        widgets = self.view.widgets
        data_types = widgets.get('ignored_data_types')
        if data_types is not None:
            data_types.object_data = {
                'ams-select2-multiple': True,
                'ams-select2-data': json.dumps(get_all_data_types(self.request))
            }
            alsoProvides(data_types, IObjectData)
        themes_target = get_parent(self.context, IThemesManagerTarget)
        if themes_target is not None:
            themes_manager = IThemesManager(themes_target)
            filter_1 = widgets.get('themes_filter_1')
            if filter_1 is not None:
                filter_1.thesaurus_name = themes_manager.thesaurus_name
                filter_1.extract_name = themes_manager.extract_name
            filter_2 = widgets.get('themes_filter_2')
            if filter_2 is not None:
                filter_2.thesaurus_name = themes_manager.thesaurus_name
                filter_2.extract_name = themes_manager.extract_name


@adapter_config(context=(ICardsWithHeadFiltersSearchResultsPortletRendererSettings, IPyAMSLayer,
                         PortletRendererPropertiesEditForm),
                provides=IFormManager)
class CardWithHeadFiltersSearchResultsPortletRendererSettingsFormManager(
        HeadFiltersSearchResultsPortletRendererSettingsFormManager):
    """Head filters search results portlet renderer settings form manager"""

    def getFields(self):
        fields = \
            Fields(ICardsWithHeadFiltersSearchResultsPortletRendererSettings).select('display_tags') + \
            Fields(ICardsWithHeadFiltersSearchResultsPortletRendererSettings).omit('display_tags')
        fields['ignored_data_types'].widgetFactory = HiddenSelect2FieldWidget
        return fields


@adapter_config(context=(IMapSearchResultsPortletRendererSettings, IPyAMSLayer,
                         PortletRendererPropertiesEditForm),
                provides=IFormManager)
class MapSearchResultsPortletRendererSettingsFormManager(DialogEditFormFormManager):
    """Search results portlet map renderer settings properties edit form manager"""

    def getFields(self):
        fields = Fields(IMapSearchResultsPortletRendererSettings).omit(
            'use_default_map_configuration')
        fields['no_use_default_map_configuration'].widgetFactory = SingleCheckBoxFieldWidget
        return fields

    def updateGroups(self):
        form = self.view
        form.add_group(NamedWidgetsGroup(form, 'configuration', form.widgets,
                                         ('no_use_default_map_configuration', ) + tuple([
                                             name
                                             for name, field in getFieldsInOrder(IMapConfiguration)
                                         ]),
                                         legend=_("Don't use default map configuration"),
                                         css_class='inner',
                                         switch=True,
                                         checkbox_switch=True,
                                         checkbox_field=IMapSearchResultsPortletRendererSettings[
                                             'no_use_default_map_configuration']))
        super().updateGroups()


@adapter_config(context=(IMapWithHeadFiltersSearchResultsPortletRendererSettings, IPyAMSLayer,
                         PortletRendererPropertiesEditForm),
                provides=IFormManager)
class MapWithHeadFiltersSearchResultsPortletRendererSettingsFormManager(DialogEditFormFormManager):
    """Search results portlet map renderer settings properties edit form manager"""

    def getFields(self):
        fields = Fields(IMapWithHeadFiltersSearchResultsPortletRendererSettings).omit(
            'use_default_map_configuration')
        fields['ignored_data_types'].widgetFactory = HiddenSelect2FieldWidget
        fields['no_use_default_map_configuration'].widgetFactory = SingleCheckBoxFieldWidget
        return fields

    def updateGroups(self):
        form = self.view
        form.add_group(NamedWidgetsGroup(form, 'configuration', form.widgets,
                                         ('no_use_default_map_configuration', ) + tuple([
                                             name
                                             for name, field in getFieldsInOrder(IMapConfiguration)
                                         ]),
                                         legend=_("Don't use default map configuration"),
                                         css_class='inner',
                                         switch=True,
                                         checkbox_switch=True,
                                         checkbox_field=IMapWithHeadFiltersSearchResultsPortletRendererSettings[
                                             'no_use_default_map_configuration']))
        super().updateGroups()

    def updateWidgets(self, prefix=None):
        super().updateWidgets(prefix)
        widgets = self.view.widgets
        data_types = widgets.get('ignored_data_types')
        if data_types is not None:
            data_types.object_data = {
                'ams-select2-multiple': True,
                'ams-select2-data': json.dumps(get_all_data_types(self.request))
            }
            alsoProvides(data_types, IObjectData)
        themes_target = get_parent(self.context, IThemesManagerTarget)
        if themes_target is not None:
            themes_manager = IThemesManager(themes_target)
            filter_1 = widgets.get('themes_filter_1')
            if filter_1 is not None:
                filter_1.thesaurus_name = themes_manager.thesaurus_name
                filter_1.extract_name = themes_manager.extract_name
            filter_2 = widgets.get('themes_filter_2')
            if filter_2 is not None:
                filter_2.thesaurus_name = themes_manager.thesaurus_name
                filter_2.extract_name = themes_manager.extract_name
