#
# Copyright (c) 2015-2021 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS_*** module

"""

from zope.interface import Interface, implementer
from zope.schema.fieldproperty import FieldProperty

from onf_website.component.location.skin.map import BaseMapMarkersRendererMixin
from onf_website.features.search.skin import AggregatedSearchResultsRenderer
from onf_website.features.search.skin.interfaces import \
    IMapSearchResultsPortletRendererSettings, \
    IMapWithHeadFiltersSearchResultsPortletRendererSettings
from pyams_content.features.search.portlet import IAggregatedPortletRenderer, \
    ISearchResultsPortletSettings
from pyams_default_theme.features.search.portlet import SearchResultsPortletBaseRenderer
from pyams_file.property import FileProperty
from pyams_gis.configuration import MapConfiguration
from pyams_gis.interfaces.configuration import IMapConfiguration
from pyams_gis.interfaces.utility import IMapManager
from pyams_portal.interfaces import IPortalContext, IPortletRenderer
from pyams_skin.layer import IPyAMSLayer
from pyams_template.template import template_config
from pyams_utils.adapter import adapter_config
from pyams_utils.factory import factory_config
from pyams_utils.registry import get_utility


__docformat__ = 'restructuredtext'

from onf_website import _


@factory_config(IMapSearchResultsPortletRendererSettings)
class MapSearchResultsPortletRendererSettings(MapConfiguration):
    """Map search results portlet renderer settings"""

    map_height = FieldProperty(
        IMapWithHeadFiltersSearchResultsPortletRendererSettings['map_height'])
    display_illustrations = FieldProperty(
        IMapWithHeadFiltersSearchResultsPortletRendererSettings['display_illustrations'])
    marker_image = FileProperty(
        IMapWithHeadFiltersSearchResultsPortletRendererSettings['marker_image'])
    display_clusters = FieldProperty(
        IMapWithHeadFiltersSearchResultsPortletRendererSettings['display_clusters'])

    _use_default_map_configuration = FieldProperty(
        IMapWithHeadFiltersSearchResultsPortletRendererSettings['use_default_map_configuration'])

    allow_pagination = False

    @property
    def use_default_map_configuration(self):
        return self._use_default_map_configuration

    @use_default_map_configuration.setter
    def use_default_map_configuration(self, value):
        self._use_default_map_configuration = value

    @property
    def no_use_default_map_configuration(self):
        return not bool(self.use_default_map_configuration)

    @no_use_default_map_configuration.setter
    def no_use_default_map_configuration(self, value):
        self.use_default_map_configuration = not bool(value)

    @property
    def configuration(self):
        if self.use_default_map_configuration:
            manager = get_utility(IMapManager)
            return IMapConfiguration(manager).get_configuration()
        return self.get_configuration()


@adapter_config(name='map',
                context=(IPortalContext, IPyAMSLayer, Interface, ISearchResultsPortletSettings),
                provides=IPortletRenderer)
@template_config(template='templates/search-map.pt', layer=IPyAMSLayer)
class MapSearchResultsPortletRenderer(BaseMapMarkersRendererMixin,
                                      SearchResultsPortletBaseRenderer):
    """Search results portlet map renderer"""

    label = _("Search results map")

    settings_interface = IMapSearchResultsPortletRendererSettings
    weight = 70

    default_page_length = 999

    def get_markers(self, results, marker_image=None):
        return super().get_markers(results, self.renderer_settings.marker_image)


#
# Search results map renderer with head filters
#

@factory_config(IMapWithHeadFiltersSearchResultsPortletRendererSettings)
@implementer(IAggregatedPortletRenderer)
class MapWithHeadFiltersSearchResultsPortletRendererSettings(MapSearchResultsPortletRendererSettings):
    """Search results portlet map with head filters renderer settings"""

    placeholder = FieldProperty(
        IMapWithHeadFiltersSearchResultsPortletRendererSettings['placeholder'])
    content_types_filter = FieldProperty(
        IMapWithHeadFiltersSearchResultsPortletRendererSettings['content_types_filter'])
    _content_types_label = FieldProperty(
        IMapWithHeadFiltersSearchResultsPortletRendererSettings['content_types_label'])
    data_types_filter = FieldProperty(
        IMapWithHeadFiltersSearchResultsPortletRendererSettings['data_types_filter'])
    ignored_data_types = FieldProperty(
        IMapWithHeadFiltersSearchResultsPortletRendererSettings['ignored_data_types'])
    _data_types_label = FieldProperty(
        IMapWithHeadFiltersSearchResultsPortletRendererSettings['data_types_label'])
    themes_filter_1 = FieldProperty(
        IMapWithHeadFiltersSearchResultsPortletRendererSettings['themes_filter_1'])
    themes_filter_1_label = FieldProperty(
        IMapWithHeadFiltersSearchResultsPortletRendererSettings['themes_filter_1_label'])
    themes_filter_1_alpha = FieldProperty(
        IMapWithHeadFiltersSearchResultsPortletRendererSettings['themes_filter_1_alpha'])
    themes_filter_2 = FieldProperty(
        IMapWithHeadFiltersSearchResultsPortletRendererSettings['themes_filter_2'])
    themes_filter_2_label = FieldProperty(
        IMapWithHeadFiltersSearchResultsPortletRendererSettings['themes_filter_2_label'])
    themes_filter_2_alpha = FieldProperty(
        IMapWithHeadFiltersSearchResultsPortletRendererSettings['themes_filter_2_alpha'])

    @property
    def content_types_label(self):
        label = self._content_types_label
        if not label:
            label = {
                'en': "Content types",
                'fr': "Types de contenus"
            }
        return label

    @content_types_label.setter
    def content_types_label(self, value):
        self._content_types_label = value

    @property
    def data_types_label(self):
        label = self._data_types_label
        if not label:
            label = {
                'en': "Data types",
                'fr': "Types de contenus"
            }
        return label

    @data_types_label.setter
    def data_types_label(self, value):
        self._data_types_label = value

    @property
    def aggregates(self):
        result = []
        if self.content_types_filter:
            result.append({
                'name': 'content_type',
                'type': 'terms',
                'params': {
                    'field': 'content_type',
                    'size': 100
                }
            })
        if self.data_types_filter:
            result.append({
                'name': 'data_type',
                'type': 'terms',
                'params': {
                    'field': 'data_type',
                    'size': 100
                }
            })
        if self.themes_filter_1 or self.themes_filter_2:
            result.append({
                'name': 'themes',
                'type': 'terms',
                'params': {
                    'field': 'themes.terms',
                    'size': 100
                }
            })
        return result


@adapter_config(name='map-filters',
                context=(IPortalContext, IPyAMSLayer, Interface, ISearchResultsPortletSettings),
                provides=IPortletRenderer)
@template_config(template='templates/search-map-filters.pt', layer=IPyAMSLayer)
class MapWithHeadFiltersSearchResultsPortletRenderer(BaseMapMarkersRendererMixin,
                                                     AggregatedSearchResultsRenderer):
    """Search results portlet map with head filters renderer"""

    label = _("Search results map with head filters")

    settings_interface = IMapWithHeadFiltersSearchResultsPortletRendererSettings
    weight = 80

    default_page_length = 999

    def get_markers(self, results, marker_image=None):
        return super().get_markers(results, self.renderer_settings.marker_image)
