#
# Copyright (c) 2015-2020 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS_*** module

"""

from zope.interface import Attribute, Interface
from zope.schema import Bool, Choice, Int, Set, TextLine

from pyams_content.shared.common.interfaces.types import ALL_DATA_TYPES_VOCABULARY
from pyams_default_theme.features.search.portlet import ISearchResultsPortletRendererBaseSettings
from pyams_file.schema import FileField
from pyams_gis.interfaces.configuration import IMapConfiguration
from pyams_i18n.schema import I18nTextLineField
from pyams_thesaurus.schema import ThesaurusTermsListField


__docformat__ = 'restructuredtext'

from onf_website import _


class IHeadFiltersSettings(Interface):
    """Head filters settings interface"""

    placeholder = I18nTextLineField(title=_("Search placeholder"),
                                    description=_("Placeholder text displayed in free search "
                                                  "input"),
                                    required=False)

    content_types_filter = Bool(title=_("Filter content types?"),
                                description=_("If 'no', content types filter will not be "
                                              "displayed"),
                                required=True,
                                default=True)

    content_types_label = I18nTextLineField(title=_("Content types label"),
                                            description=_("Label used for content types "
                                                          "selection box"),
                                            required=False)

    data_types_filter = Bool(title=_("Filter data types?"),
                             description=_("If 'yes', content data types filter will be "
                                           "displayed"),
                             required=True,
                             default=False)

    ignored_data_types = Set(title=_("Ignored data types"),
                             description=_("These data types, if any, will be excluded from "
                                           "selection list"),
                             value_type=Choice(vocabulary=ALL_DATA_TYPES_VOCABULARY),
                             required=False)

    data_types_label = I18nTextLineField(title=_("Data types label"),
                                         description=_("Label used for data types selection box"),
                                         required=False)

    themes_filter_1 = ThesaurusTermsListField(title=_("First filter themes"),
                                              description=_("List of themes available in the "
                                                            "first results filter"),
                                              required=False)

    themes_filter_1_label = I18nTextLineField(title=_("First filter label"),
                                              description=_("Label displayed in the first themes "
                                                            "filter when no option is selected"),
                                              required=False)

    themes_filter_1_alpha = Bool(title=_("Force alphabetical order"),
                                 description=_("Force alphabetical order instead of count order"),
                                 required=True,
                                 default=False)

    themes_filter_2 = ThesaurusTermsListField(title=_("Second filter themes"),
                                              description=_("List of themes available in the "
                                                            "second results filter"),
                                              required=False)

    themes_filter_2_label = I18nTextLineField(title=_("Second filter label"),
                                              description=_("Label displayed in the second "
                                                            "themes filter when no option is "
                                                            "selected"),
                                              required=False)

    themes_filter_2_alpha = Bool(title=_("Force alphabetical order"),
                                 description=_("Force alphabetical order instead of count order"),
                                 required=True,
                                 default=False)


class IHeadFiltersSearchResultsPortletRendererSettings(
        ISearchResultsPortletRendererBaseSettings, IHeadFiltersSettings):
    """Head filtered search results portlet renderer settings"""


class ICardsWithHeadFiltersSearchResultsPortletRendererSettings(
        IHeadFiltersSearchResultsPortletRendererSettings):
    """Cards with head filters search results portlet renderer settings"""

    main_css_class = TextLine(title=_("Main CSS class"),
                              description=_("CSS class associated with the main block"),
                              required=True,
                              default='col-lg-10 col-lg-offset-1')

    inner_css_class = TextLine(title=_("Inner CSS class"),
                               description=_("CSS class associated with inner blocks"),
                               required=True,
                               default='col-sm-6 col-md-4')

    display_tags = Bool(title=_("Display content types"),
                        required=True,
                        default=True)


class IMapSearchResultsPortletRendererSettings(IMapConfiguration):
    """Search results portlet map renderer settings"""

    no_use_default_map_configuration = Bool(title=_("Don't use default configuration?"),
                                            required=True,
                                            default=False)

    use_default_map_configuration = Bool(title=_("Use default configuration?"),
                                         required=True,
                                         default=True)

    configuration = Attribute("Map configuration")

    map_height = Int(title=_("Map height"),
                     description=_("Map height, in pixels"),
                     required=True,
                     default=400)

    display_illustrations = Bool(title=_("Display illustrations?"),
                                 description=_("If 'yes', illustrations associated to "
                                               "search results will be displayed in "
                                               "markers tooltips"),
                                 required=True,
                                 default=True)

    marker_image = FileField(title=_("Custom marker image"),
                             description=_("This image will replace default marker..."),
                             required=False)

    display_clusters = Bool(title=_("Display markers clusters"),
                            description=_("If 'yes', location markers will be grouped in "
                                          "clusters at higher map scale"),
                            required=True,
                            default=True)


class IMapWithHeadFiltersSearchResultsPortletRendererSettings(
        IMapSearchResultsPortletRendererSettings, IHeadFiltersSettings):
    """Search results portlet map renderer with head filters settings"""
