#
# Copyright (c) 2015-2023 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS_*** module

"""

import logging

from firebase_admin.messaging import MulticastMessage, Notification, send_multicast
from pyramid.events import subscriber
from pyramid.interfaces import ILocalizer
from pyramid.settings import asbool
from transaction.interfaces import ITransactionManager

from onf_website.component.location import ILocationInfo
from onf_website.features.firebase import IFirebaseSubscriptions, check_firebase_app
from onf_website.features.firebase.interfaces import ACCOUNT_DRY_RUN_KEY
from pyams_content.root import ISiteRoot
from pyams_content.shared.common import IWfSharedContent
from pyams_content.shared.common.interfaces import IWfSharedContentNotificationEvent
from pyams_i18n.interfaces import II18n, INegotiator
from pyams_sequence.interfaces import ISequentialIdInfo
from pyams_utils.list import paginate
from pyams_utils.registry import get_global_registry
from pyams_utils.traversing import get_parent
from pyams_workflow.interfaces import IWorkflow, IWorkflowState

from onf_website import _


LOGGER = logging.getLogger('PyAMS (Firebase)')

MULTICAST_PAGE_SIZE = 500


def notify(status, firebase_ids, title, body, data, dry_run):
    """Notify given Firebase IDs with message data"""
    if not status:
        return
    check_firebase_app()
    LOGGER.info('>>> sending message to %d recipients: ', len(firebase_ids))
    LOGGER.debug('  > notification: %s -- %s', title, body)
    LOGGER.debug('  > data: %s', str(data))
    for tokens in paginate(firebase_ids, MULTICAST_PAGE_SIZE):
        message = MulticastMessage(tokens=tokens,
                                   notification=Notification(title, body),
                                   data=data)
        response = send_multicast(message,
                                  dry_run=dry_run)
        LOGGER.info('<<< status: %d message(s) - %d error(s)', response.success_count, response.failure_count)


@subscriber(IWfSharedContentNotificationEvent)
def handle_content_notifications(event):
    """Handle content notifications"""
    content = event.object
    workflow = IWorkflow(content)
    workflow_state = IWorkflowState(content)
    if workflow_state.state not in workflow.published_states:
        return
    location = ILocationInfo(content, None)
    if location is None:
        return
    forests_ids = location.forests_index
    if not forests_ids:
        return
    request = event.request
    if request is None:
        root = get_parent(content, ISiteRoot)
    else:
        root = request.root
    subscriptions = IFirebaseSubscriptions(root, None)
    if subscriptions is None:
        return
    content_type = IWfSharedContent(content).content_type
    firebase_ids = list(subscriptions.get_subscriptions(content_type, forests_ids))
    if not firebase_ids:
        return
    # get translater
    registry = get_global_registry()
    translate = None
    if request is not None:
        translate = request.localizer.translate
    else:
        sm = root.getSiteManager()
        negotiator = sm.queryUtility(INegotiator)
        if negotiator is not None:
            localizer = registry.queryUtility(ILocalizer, name=negotiator.server_language)
            if localizer is not None:
                translate = localizer.translate
    message_title = translate(_("New publication"))
    message_body = II18n(content).query_attribute('title', request=request)
    message_data = {
        'content_type': content_type,
        'oid': ISequentialIdInfo(content).hex_oid
    }
    ITransactionManager(content).get() \
        .addAfterCommitHook(notify,
                            kws={
                                'firebase_ids': firebase_ids,
                                'title': message_title,
                                'body': message_body,
                                'data': message_data,
                                'dry_run': asbool(registry.settings.get(ACCOUNT_DRY_RUN_KEY, False))
                            })
