#
# Copyright (c) 2015-2023 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS_*** module

"""

import firebase_admin

from persistent import Persistent
from persistent.mapping import PersistentMapping
from zope.container.btree import BTreeContainer
from zope.container.contained import Contained
from zope.lifecycleevent import ObjectModifiedEvent
from zope.schema.fieldproperty import FieldProperty

from onf_website.features.firebase.interfaces import ACCOUNT_KEY_PATH_KEY, FIREBASE_SUBSCRIPTIONS_KEY, \
    IFirebaseSubscription, IFirebaseSubscriptions
from pyams_content.root import ISiteRoot
from pyams_utils.adapter import adapter_config, get_annotation_adapter
from pyams_utils.factory import create_object, factory_config
from pyams_utils.registry import get_global_registry


def check_firebase_app():
    """Firebase application initialization"""
    try:
        _app = firebase_admin.get_app()
    except ValueError:
        registry = get_global_registry()
        settings = registry.settings
        key_path = settings.get(ACCOUNT_KEY_PATH_KEY)
        if key_path:
            cred = firebase_admin.credentials.Certificate(key_path)
            firebase_admin.initialize_app(cred)


@factory_config(IFirebaseSubscription)
class FirebaseSubscription(Persistent, Contained):
    """Subscription persistent class"""

    firebase_id = FieldProperty(IFirebaseSubscription['firebase_id'])
    content_types = FieldProperty(IFirebaseSubscription['content_types'])

    def __init__(self, firebase_id):
        super().__init__()
        self.firebase_id = firebase_id
        self.content_types = PersistentMapping()

    def add(self, content_type, forests):
        self.content_types[content_type] = (self.content_types.get(content_type) or set()) | set(forests)
        registry = get_global_registry()
        registry.notify(ObjectModifiedEvent(self))

    def drop(self, content_type, forests):
        if content_type:
            if content_type not in self.content_types:
                return bool(self.content_types)
            forests_ids = self.content_types[content_type]
            if not forests_ids:
                return bool(self.content_types)
            if forests:
                forests_ids = forests_ids - set(forests or ())
                if forests_ids:
                    self.content_types[content_type] = forests_ids
                else:
                    del self.content_types[content_type]
            else:
                del self.content_types[content_type]
        else:
            self.content_types.clear()
        registry = get_global_registry()
        registry.notify(ObjectModifiedEvent(self))
        return bool(self.content_types)


_marker = dict()


@factory_config(IFirebaseSubscriptions)
class FirebaseSubscriptions(BTreeContainer):
    """Firebase subscriptions container"""

    def __init__(self):
        super().__init__()
        self.by_forest = PersistentMapping()

    def add_subscriptions(self, firebase_id, content_type, forests):
        """Add subscriptions for provided Firebase ID"""
        subscription = self.get(firebase_id)
        if subscription is None:
            subscription = create_object(IFirebaseSubscription,
                                         firebase_id=firebase_id,
                                         notify=True)
        subscription.add(content_type, forests)
        self[firebase_id] = subscription
        for forest in forests:
            values = self.by_forest.setdefault(forest, PersistentMapping()).setdefault(content_type, set())
            if firebase_id not in values:
                values.add(firebase_id)
                self.by_forest[forest][content_type] = values

    def drop_subscriptions(self, firebase_id, content_type=None, forests=None):
        """Drop subscriptions for provided Firebase ID"""
        subscription = self.get(firebase_id)
        if subscription is None:
            return
        if not subscription.drop(content_type, forests):
            del self[firebase_id]
        for forest_id, subscription_types in list(self.by_forest.items()):
            if forests and (forest_id not in forests):
                continue
            for subscription_type, firebase_ids in list(subscription_types.items()):
                if content_type and (content_type != subscription_type):
                    continue
                if firebase_id in firebase_ids:
                    firebase_ids.remove(firebase_id)
                    if firebase_ids:
                        self.by_forest[forest_id][subscription_type] = firebase_ids
                    else:
                        del self.by_forest[forest_id][subscription_type]
            if not self.by_forest[forest_id]:
                del self.by_forest[forest_id]

    def update_firebase_id(self, old_firebase_id, new_firebase_id):
        """Replace old Firebase ID with new one"""
        old_subscription = self.get(old_firebase_id)
        if old_subscription is None:
            return
        new_subscription = self.get(new_firebase_id)
        if new_subscription is not None:
            return
        for content_type, forests in old_subscription.content_types.items():
            self.add_subscriptions(new_firebase_id, content_type, forests)
        self.drop_subscriptions(old_firebase_id)
        return new_subscription

    def get_subscriptions(self, content_type, forests):
        """Get Firebase IDs for provided subscriptions params"""
        return set([
            firebase_id
            for forest in (forests or ())
            for firebase_id in self.by_forest.get(forest, {}).get(content_type, ())
            if (firebase_id in self) and (forest in self[firebase_id].content_types.get(content_type, ()))
        ])


@adapter_config(required=ISiteRoot,
                provides=IFirebaseSubscriptions)
def firebase_subscriptions(context):
    """Firebase subscriptions adapter"""
    return get_annotation_adapter(context, FIREBASE_SUBSCRIPTIONS_KEY, IFirebaseSubscriptions)
