# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from transaction.interfaces import ITransactionManager
from z3c.form import button
from z3c.form.interfaces import INPUT_MODE
from zope.interface import alsoProvides, implementer

from onf_website.component.paragraph.interfaces.contact import IContactParagraph
from pyams_content.component.paragraph.interfaces import IBaseParagraph, IParagraphContainerTarget
from pyams_content.component.paragraph.interfaces.contact import IContactParagraph as IContactParagraphBase
from pyams_content.component.paragraph.zmi import BaseParagraphAJAXAddForm, BaseParagraphAJAXEditForm, \
    IParagraphInnerEditFormButtons, get_json_paragraph_refresh_event
from pyams_content.component.paragraph.zmi.contact import ContactParagraphAddForm as BaseContactParagraphAddForm, \
    ContactParagraphPropertiesEditForm as BaseContactParagraphPropertiesEditForm
from pyams_content.component.paragraph.zmi.interfaces import IParagraphInnerEditor
from pyams_content.interfaces import MANAGE_CONTENT_PERMISSION
from pyams_form.form import ajax_config
from pyams_form.interfaces.form import IInnerForm
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.event import get_json_form_refresh_event
from pyams_skin.layer import IPyAMSLayer
from pyams_utils.adapter import adapter_config
from pyams_utils.interfaces.data import IObjectData


@pagelet_config(name='add-contact-paragraph.html', context=IParagraphContainerTarget, layer=IPyAMSLayer,
                permission=MANAGE_CONTENT_PERMISSION)
@ajax_config(name='add-contact-paragraph.json', context=IParagraphContainerTarget, layer=IPyAMSLayer,
             base=BaseParagraphAJAXAddForm)
class ContactParagraphAddForm(BaseContactParagraphAddForm):
    """Contact paragraph add form"""

    content_interface = IContactParagraph

    @property
    def fields(self):
        return super(ContactParagraphAddForm, self).fields \
            .select('title', 'name', 'charge', 'company', 'address', 'phone_number',
                    'contact_email', 'contact_form', 'photo', 'gps_location', 'city', 'renderer')

    def updateWidgets(self, prefix=None):
        super(ContactParagraphAddForm, self).updateWidgets(prefix)
        if 'phone_number' in self.widgets:
            widget = self.widgets['phone_number']
            widget.object_data = {
                'input-mask': '99 99 99 99 99'
            }
            alsoProvides(widget, IObjectData)


@pagelet_config(name='properties.html', context=IContactParagraph, layer=IPyAMSLayer,
                permission=MANAGE_CONTENT_PERMISSION)
@ajax_config(name='properties.json', context=IContactParagraph, layer=IPyAMSLayer, base=BaseParagraphAJAXEditForm)
class ContactParagraphPropertiesEditForm(BaseContactParagraphPropertiesEditForm):
    """Contact paragraph properties edit form"""

    content_interface = IContactParagraph

    @property
    def fields(self):
        return super(ContactParagraphPropertiesEditForm, self).fields \
            .select('title', 'name', 'charge', 'company', 'address', 'phone_number',
                    'contact_email', 'contact_form', 'photo', 'gps_location', 'city', 'renderer')

    def updateWidgets(self, prefix=None):
        super(ContactParagraphPropertiesEditForm, self).updateWidgets(prefix)
        if 'phone_number' in self.widgets:
            widget = self.widgets['phone_number']
            widget.object_data = {
                'input-mask': '99 99 99 99 99'
            }
            alsoProvides(widget, IObjectData)


@adapter_config(context=(IContactParagraph, IPyAMSLayer), provides=IParagraphInnerEditor)
@ajax_config(name='inner-properties.json', context=IContactParagraph, layer=IPyAMSLayer,
             permission=MANAGE_CONTENT_PERMISSION, base=BaseParagraphAJAXEditForm)
@implementer(IInnerForm)
class ContactParagraphInnerEditForm(ContactParagraphPropertiesEditForm):
    legend = None

    @property
    def buttons(self):
        if self.mode == INPUT_MODE:
            return button.Buttons(IParagraphInnerEditFormButtons)
        else:
            return button.Buttons()

    def get_ajax_output(self, changes):
        output = super(self.__class__, self).get_ajax_output(changes)
        updated = changes.get(IBaseParagraph, ())
        if 'title' in updated:
            output.setdefault('events', []).append(get_json_paragraph_refresh_event(self.context, self.request))
        updated = changes.get(IContactParagraphBase, ())
        if ('photo' in updated) or ('renderer' in updated):
            # we have to commit transaction to be able to handle blobs...
            ITransactionManager(self.context).get().commit()
            output.setdefault('events', []).append(get_json_form_refresh_event(self.context, self.request,
                                                                               ContactParagraphInnerEditForm))
        return output

    def updateWidgets(self, prefix=None):
        super(ContactParagraphInnerEditForm, self).updateWidgets(prefix)
        if 'phone_number' in self.widgets:
            widget = self.widgets['phone_number']
            widget.object_data = {
                'input-mask': '99 99 99 99 99'
            }
            alsoProvides(widget, IObjectData)
