#
# Copyright (c) 2008-2017 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from zope.schema.fieldproperty import FieldProperty

from onf_website.component.paragraph.skin.interfaces.frame import ACTION_FRAME_PARAGRAPH_RENDERER_SETTINGS_KEY, \
    IActionFrameParagraphRendererSettings, \
    IUnborderedFrameParagraphRendererSettings, \
    IUnborderedLeftFrameParagraphRendererSettings, IUnborderedRightFrameParagraphRendererSettings
from pyams_content.component.paragraph.interfaces.frame import IFrameParagraph
from pyams_content.features.renderer.interfaces import IContentRenderer
from pyams_default_theme.component.paragraph.frame import BaseFrameParagraphRendererSettings, \
    DefaultFrameParagraphRenderer, FRAME_PARAGRAPH_RENDERER_SETTINGS_KEY, LEFT_FRAME_PARAGRAPH_RENDERER_SETTINGS_KEY, \
    LateralFrameParagraphRendererSettings, LeftFrameParagraphRenderer, RIGHT_FRAME_PARAGRAPH_RENDERER_SETTINGS_KEY, \
    RightFrameParagraphRenderer
from pyams_sequence.reference import get_reference_target
from pyams_skin.layer import IPyAMSLayer
from pyams_template.template import template_config
from pyams_utils.adapter import adapter_config, get_annotation_adapter
from pyams_utils.factory import factory_config

from onf_website import _
from pyams_utils.url import canonical_url


#
# Un-bordered framed text paragraph renderer settings
#

@factory_config(IUnborderedFrameParagraphRendererSettings)
class UnborderedFrameParagraphRendererSettings(BaseFrameParagraphRendererSettings):
    """Un-bordered framed paragraph renderer settings"""


@adapter_config(context=IFrameParagraph,
                provides=IUnborderedFrameParagraphRendererSettings)
def unbordered_frame_paragraph_renderer_settings_factory(context):
    """Framed paragraph default renderer settings factory"""
    return get_annotation_adapter(context, FRAME_PARAGRAPH_RENDERER_SETTINGS_KEY + '.unbordered',
                                  IUnborderedFrameParagraphRendererSettings)


@factory_config(IUnborderedLeftFrameParagraphRendererSettings)
class UnborderedLeftFrameParagraphRendererSettings(LateralFrameParagraphRendererSettings):
    """Un-bordered left framed paragraph lateral renderer settings"""


@adapter_config(context=IFrameParagraph,
                provides=IUnborderedLeftFrameParagraphRendererSettings)
def unbordered_left_lateral_frame_paragraph_renderer_settings_factory(context):
    """Left framed paragraph renderer settings factory"""
    return get_annotation_adapter(context, LEFT_FRAME_PARAGRAPH_RENDERER_SETTINGS_KEY + '.unbordered',
                                  IUnborderedLeftFrameParagraphRendererSettings)


@factory_config(IUnborderedRightFrameParagraphRendererSettings)
class UnborderedRightFrameParagraphRendererSettings(LateralFrameParagraphRendererSettings):
    """Un-bordered right framed paragraph lateral renderer settings"""


@adapter_config(context=IFrameParagraph,
                provides=IUnborderedRightFrameParagraphRendererSettings)
def unbordered_right_lateral_frame_paragraph_renderer_settings_factory(context):
    """Right framed paragraph lateral renderer settings factory"""
    return get_annotation_adapter(context, RIGHT_FRAME_PARAGRAPH_RENDERER_SETTINGS_KEY + '.unbordered',
                                  IUnborderedRightFrameParagraphRendererSettings)


#
# Framed text paragraph renderers
#

@adapter_config(name='default',
                context=(IFrameParagraph, IPyAMSLayer),
                provides=IContentRenderer)
@template_config(template='templates/frame-default.pt',
                 layer=IPyAMSLayer)
@template_config(name='PARA:title-left',
                 template='templates/frame-title-left.pt',
                 layer=IPyAMSLayer)
class ONFDefaultFrameParagraphRenderer(DefaultFrameParagraphRenderer):
    """Framed text paragraph default renderer"""

    label = _("ONF: Default centered frame")
    weight = 1


@adapter_config(name='unbordered',
                context=(IFrameParagraph, IPyAMSLayer),
                provides=IContentRenderer)
@template_config(template='templates/frame-unbordered.pt',
                 layer=IPyAMSLayer)
class ONFUnborderedFrameParagraphRenderer(DefaultFrameParagraphRenderer):
    """Framed text paragraph default renderer"""

    label = _("ONF: Un-bordered centered frame")
    weight = 2

    settings_interface = IUnborderedFrameParagraphRendererSettings


@adapter_config(name='left',
                context=(IFrameParagraph, IPyAMSLayer),
                provides=IContentRenderer)
@template_config(template='templates/frame-left.pt',
                 layer=IPyAMSLayer)
class ONFLeftFrameParagraphRenderer(LeftFrameParagraphRenderer):
    """Left framed text paragraph renderer"""

    label = _("ONF: Small bordered frame on the left")
    weight = 3


@adapter_config(name='unbordered-left',
                context=(IFrameParagraph, IPyAMSLayer),
                provides=IContentRenderer)
@template_config(template='templates/frame-unbordered-left.pt',
                 layer=IPyAMSLayer)
class ONFUnborderedLeftFrameParagraphRenderer(LeftFrameParagraphRenderer):
    """Left un-bordered framed paragraph renderer"""

    label = _("ONF: Un-bordered frame on the left")
    weight = 4

    settings_interface = IUnborderedLeftFrameParagraphRendererSettings


@adapter_config(name='right',
                context=(IFrameParagraph, IPyAMSLayer),
                provides=IContentRenderer)
@template_config(template='templates/frame-right.pt',
                 layer=IPyAMSLayer)
class ONFRightFrameParagraphRenderer(RightFrameParagraphRenderer):
    """Right framed text paragraph renderer"""

    label = _("ONF: Small bordered frame on the right")
    weight = 5


@adapter_config(name='unbordered-right',
                context=(IFrameParagraph, IPyAMSLayer),
                provides=IContentRenderer)
@template_config(template='templates/frame-unbordered-right.pt',
                 layer=IPyAMSLayer)
class ONFUnborderedRightFrameParagraphRenderer(RightFrameParagraphRenderer):
    """Right un-bordered framed paragraph renderer"""

    label = _("ONF: Un-bordered frame on the right")
    weight = 6

    settings_interface = IUnborderedRightFrameParagraphRendererSettings


@factory_config(IActionFrameParagraphRendererSettings)
class ActionFrameParagraphRendererSettings(BaseFrameParagraphRendererSettings):
    """Call-to-action framed paragraph renderer settings"""

    button_label = FieldProperty(IActionFrameParagraphRendererSettings['button_label'])
    button_target_reference = FieldProperty(
        IActionFrameParagraphRendererSettings['button_target_reference'])
    button_target_url = FieldProperty(
        IActionFrameParagraphRendererSettings['button_target_url'])


@adapter_config(context=IFrameParagraph,
                provides=IActionFrameParagraphRendererSettings)
def action_frame_paragraph_renderer_settings_factory(context):
    """Action framed paragraph renderer settings factory"""
    return get_annotation_adapter(context, ACTION_FRAME_PARAGRAPH_RENDERER_SETTINGS_KEY,
                                  IActionFrameParagraphRendererSettings)


@adapter_config(name='action',
                context=(IFrameParagraph, IPyAMSLayer),
                provides=IContentRenderer)
@template_config(template='templates/frame-action.pt',
                 layer=IPyAMSLayer)
class ONFActionFrameParagraphRenderer(DefaultFrameParagraphRenderer):
    """Call-to-action framed paragraph renderer"""

    label = _("ONF: call-to-action frame")
    weight = 7

    settings_interface = IActionFrameParagraphRendererSettings

    @property
    def button_target_url(self):
        settings = self.settings
        target = get_reference_target(settings.button_target_reference)
        if target is not None:
            return canonical_url(target, self.request)
        return settings.button_target_url or None
