#
# Copyright (c) 2015-2022 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS_*** module

"""

__docformat__ = 'restructuredtext'

from pyams_content.component.map.interfaces import have_gis
if have_gis:

    from zope.schema.fieldproperty import FieldProperty

    from onf_website.component.location import ILocationInfo, ILocationTarget
    from onf_website.component.map.interfaces import IMapParagraph
    from pyams_content.component.paragraph import BaseParagraphFactory
    from pyams_content.component.paragraph.interfaces import IParagraphFactory
    from pyams_content.component.map.interfaces import MAP_PARAGRAPH_TYPE
    from pyams_content.component.map.paragraph import JSONMapParagraphConverter as JSONMapParagraphConverterBase, \
        MapParagraph as BaseMapParagraph
    from pyams_content.features.json.interfaces import IJSONConverter
    from pyams_gis.interfaces import WGS84
    from pyams_skin.layer import IPyAMSLayer
    from pyams_utils.adapter import adapter_config
    from pyams_utils.factory import factory_config
    from pyams_utils.registry import utility_config
    from pyams_utils.traversing import get_parent

    from onf_website import _


    @factory_config(IMapParagraph)
    class MapParagraph(BaseMapParagraph):
        """Map paragraph"""

        _use_context_location = FieldProperty(IMapParagraph['use_context_location'])

        @property
        def use_context_location(self):
            return self._use_context_location

        @use_context_location.setter
        def use_context_location(self, value):
            self._use_context_location = value

        @property
        def no_use_context_location(self):
            return not bool(self.use_context_location)

        @no_use_context_location.setter
        def no_use_context_location(self, value):
            self.use_context_location = not bool(value)


    @utility_config(name=MAP_PARAGRAPH_TYPE, provides=IParagraphFactory)
    class MapParagraphFactory(BaseParagraphFactory):
        """Map paragraph factory"""

        name = _("Location map")
        content_type = MapParagraph
        secondary_menu = True


    @adapter_config(required=(IMapParagraph, IPyAMSLayer),
                    provides=IJSONConverter)
    class JSONMapParagraphConverter(JSONMapParagraphConverterBase):
        """JSON map paragraph converter"""

        def convert_content(self, params):
            result = super().convert_content(params)
            if self.context.use_context_location:
                parent = get_parent(self.context, ILocationTarget)
                if parent is not None:
                    location = ILocationInfo(self.context, None)
                    if (location is not None) and location.gps_location:
                        srid = params.get('srid', WGS84)
                        try:
                            coords = location.gps_location.get_coordinates(srid)
                            result['gps_location'] = {
                                'longitude': float(coords[0]),
                                'latitude': float(coords[1]),
                                'srid': srid
                            }
                        except NotImplementedError:
                            pass
                    else:
                        result.pop('gps_location', None)
            return result
