#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from z3c.form import button, field
from z3c.form.browser.checkbox import SingleCheckBoxFieldWidget
from z3c.form.interfaces import INPUT_MODE
from zope.dublincore.interfaces import IZopeDublinCore
from zope.interface import Interface, alsoProvides, implementer
from zope.lifecycleevent import Attributes, ObjectModifiedEvent

from onf_website.component.location.interfaces import ILocationGPSInfo, ILocationInfo, \
    ILocationInseeInfo, ILocationLabelInfo, ILocationLocationInfo, ILocationManagerSettings, \
    ILocationManagerTarget, ILocationStructureInfo, ILocationTarget, LOCATION_SETTINGS
from pyams_content.interfaces import MANAGE_CONTENT_PERMISSION
from pyams_content.shared.common.zmi import WfSharedContentHeaderAdapter
from pyams_form.form import ajax_config
from pyams_form.group import NamedWidgetsGroup
from pyams_form.help import FormHelp
from pyams_form.interfaces.form import IFormHelp, IInnerSubForm, IWidgetForm
from pyams_form.schema import ResetButton
from pyams_gis.zmi.interfaces import IMapHeaderViewletManager
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.event import get_json_form_refresh_event
from pyams_skin.interfaces import IInnerPage, IPageHeader
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.viewlet.menu import MenuDivider, MenuItem
from pyams_template.template import template_config
from pyams_utils.adapter import adapter_config
from pyams_utils.interfaces import VIEW_SYSTEM_PERMISSION
from pyams_utils.traversing import get_parent
from pyams_viewlet.viewlet import Viewlet, viewlet_config
from pyams_zmi.form import AdminEditForm, InnerAdminEditForm
from pyams_zmi.interfaces.menu import IPropertiesMenu
from pyams_zmi.layer import IAdminLayer


__docformat__ = 'restructuredtext'

from onf_website import _


@viewlet_config(name='location.divider',
                context=ILocationTarget, layer=IAdminLayer,
                manager=IPropertiesMenu, weight=309,
                permission=VIEW_SYSTEM_PERMISSION)
class LocationMenuDivider(MenuDivider):
    """Location menu divider"""


@viewlet_config(name='location.menu',
                context=ILocationTarget, layer=IAdminLayer,
                manager=IPropertiesMenu, weight=310,
                permission=VIEW_SYSTEM_PERMISSION)
class LocationMenu(MenuItem):
    """Location menu"""

    label = _("Location...")
    icon_class = 'fa-map-marker'
    url = '#location.html'


class ILocationEditGroup(Interface):
    """Inner location edit group"""


class ILocationEditButtons(Interface):
    """Location form buttons"""

    reset = ResetButton(name='reset', title=_("Reset"))
    submit = button.Button(name='submit', title=_("Submit"))


@pagelet_config(name='location.html',
                context=ILocationTarget, layer=IPyAMSLayer,
                permission=VIEW_SYSTEM_PERMISSION)
@ajax_config(name='location.json',
             context=ILocationTarget, layer=IPyAMSLayer,
             permission=MANAGE_CONTENT_PERMISSION,
             implementer=ILocationEditGroup)
@implementer(IWidgetForm, IInnerPage)
class LocationEditForm(AdminEditForm):
    """Location edit form"""

    legend = _("Content location")

    @property
    def fields(self):
        if ILocationInfo(self.context).can_inherit:
            fields = field.Fields(ILocationInfo).select('no_inherit')
            fields['no_inherit'].widgetFactory = SingleCheckBoxFieldWidget
        else:
            fields = field.Fields(Interface)
        return fields

    @property
    def buttons(self):
        if self.mode == INPUT_MODE:
            return button.Buttons(ILocationEditButtons)
        else:
            return button.Buttons(Interface)

    def updateGroups(self):
        if ILocationInfo(self.context).can_inherit:
            group = NamedWidgetsGroup(self, 'location', self.widgets,
                                      ('no_inherit', ),
                                      legend=_("Don't inherit parent location"),
                                      css_class='inner',
                                      switch=True,
                                      checkbox_switch=True,
                                      checkbox_mode='disable',
                                      checkbox_field=ILocationInfo['no_inherit'])
        else:
            group = NamedWidgetsGroup(self, 'location', self.widgets, (), css_class='inner')
        alsoProvides(group, ILocationEditGroup)
        self.add_group(group)
        super().updateGroups()

    def get_ajax_output(self, changes):
        if changes:
            location = ILocationInfo(self.getContent())
            descriptions = []
            for interface, names in changes.items():
                descriptions.append(Attributes(interface, *names))
            self.request.registry.notify(ObjectModifiedEvent(location, *descriptions))
        output = super(self.__class__, self).get_ajax_output(changes)
        if changes:
            output.setdefault('events', []).append(
                get_json_form_refresh_event(
                    self.context, self.request,
                    lambda context, request: LocationLabelEditSubform(context, request, self)))
        return output


class LocationInnerEditForm(InnerAdminEditForm):
    """Location inner edit form"""

    legend = None
    name = None

    def __new__(cls, context, request, view=None):
        manager = get_parent(context, ILocationManagerTarget)
        if manager is None:
            return None
        settings = ILocationManagerSettings(manager).get(cls.name)
        if (not settings) or (not settings.visible):
            return None
        return InnerAdminEditForm.__new__(cls)

    @property
    def main_group_legend(self):
        return LOCATION_SETTINGS.get(self.name)

    main_group_class = 'inner switcher no-y-padding'

    def update(self):
        super().update()
        group = self.groups[0]
        group.switch = True
        context_dc = IZopeDublinCore(self.context)
        location_dc = IZopeDublinCore(ILocationInfo(self.context))
        if location_dc.modified and (location_dc.modified > (location_dc.created or
                                                             context_dc.modified)):
            group.display_mode = 'auto'
        else:
            manager = get_parent(self.context, ILocationManagerTarget)
            setting = ILocationManagerSettings(manager)[self.name]
            group.display_mode = 'always' if setting.unfolded else 'auto'


@adapter_config(name='coordinates',
                context=(ILocationTarget, IPyAMSLayer, ILocationEditGroup),
                provides=IInnerSubForm)
class LocationCoordinatesEditSubform(LocationInnerEditForm):
    """Location coordinates edit form"""

    name = 'gps'

    fields = field.Fields(ILocationGPSInfo)
    weight = 1

    def updateWidgets(self, prefix=None):
        super().updateWidgets(prefix)
        if 'gps_location' in self.widgets:
            widget = self.widgets['gps_location']
            widget.label = ''
            widget.label_css_class = 'col-md-hide'
            widget.input_css_class = 'col-md-12'
            widget.fieldset_class = 'no-y-padding padding-x-5'


@adapter_config(name='location',
                context=(ILocationTarget, IPyAMSLayer, ILocationEditGroup),
                provides=IInnerSubForm)
class LocationLocationEditSubform(LocationInnerEditForm):
    """Location main edit form"""

    name = 'location'

    fields = field.Fields(ILocationLocationInfo)
    fields['all_departments_forests'].widgetFactory = SingleCheckBoxFieldWidget
    fields['domanials_only'].widgetFactory = SingleCheckBoxFieldWidget
    fields['communals_only'].widgetFactory = SingleCheckBoxFieldWidget
    weight = 2

    def updateWidgets(self, prefix=None):
        super().updateWidgets(prefix)
        translate = self.request.localizer.translate
        widget = self.widgets.get('forests')
        if widget is not None:
            widget.after_widget_notice = '<span class="text-danger">{0}</span>'.format(
                translate(_("If GPS coordinates given above tie in a forest, please reference "
                            "it here")))
            if 'all_departments_forests' in self.widgets:
                widget.after_widget_notice += '<br /><br />' + translate(_("...of departments "
                                                                           "selected below:"))
        widget = self.widgets.get('all_departments_forests')
        if widget is not None:
            widget.group_css_class = 'form-group col-md-6 no-y-padding'
            widget.label_css_class = 'control-label col-md-6'
            widget.input_css_class = 'col-md-6'
            widget.widget_css_class = 'input padding-left-20'
        widget = self.widgets.get('domanials_only')
        if widget is not None:
            widget.group_css_class = 'form-group col-md-3 no-y-padding'
            widget.label_css_class = 'control-label col-md-1'
            widget.input_css_class = 'col-md-11'
            widget.widget_css_class = 'input padding-left-20'
        widget = self.widgets.get('communals_only')
        if widget is not None:
            widget.group_css_class = 'form-group col-md-3 no-y-padding'
            widget.label_css_class = 'control-label col-md-1'
            widget.input_css_class = 'col-md-11'
            widget.widget_css_class = 'input padding-left-20'
        widget = self.widgets.get('cities')
        if widget is not None:
            widget.after_widget_notice = '<span class="text-danger">{0}</span>'.format(
                translate(_("If GPS coordinates given above tie in a city, please reference "
                            "it here")))


@adapter_config(name='insee',
                context=(ILocationTarget, IPyAMSLayer, ILocationEditGroup),
                provides=IInnerSubForm)
class LocationInseeEditSubform(LocationInnerEditForm):
    """Location address edit form"""

    name = 'insee'

    fields = field.Fields(ILocationInseeInfo)
    weight = 3


@adapter_config(name='structure',
                context=(ILocationTarget, IPyAMSLayer, ILocationEditGroup),
                provides=IInnerSubForm)
class LocationStructureEditSubform(LocationInnerEditForm):
    """Location structure edit form"""

    name = 'structure'

    fields = field.Fields(ILocationStructureInfo)
    weight = 4


@adapter_config(name='label',
                context=(ILocationTarget, IPyAMSLayer, ILocationEditGroup),
                provides=IInnerSubForm)
class LocationLabelEditSubform(LocationInnerEditForm):
    """Location label edit form"""

    name = 'label'

    fields = field.Fields(ILocationLabelInfo)
    weight = 5


@adapter_config(context=(ILocationTarget, IAdminLayer, LocationEditForm),
                provides=IPageHeader)
class LocationEditFormHeader(WfSharedContentHeaderAdapter):
    """Shared content location header adapter"""

    icon_class = 'fa fa-fw fa-map-marker'


@adapter_config(context=(ILocationTarget, IAdminLayer, LocationEditForm),
                provides=IFormHelp)
class LocationEditFormHelp(FormHelp):
    """Location edit form help"""

    message = _("Any content about a given territory location should be localized.\n\n"
                "**WARNING: only define the most relevant descriptors, as precise as possible** "
                "(for example, if the content concerns a forest, don't define the department or "
                "the organization).")
    message_format = 'rest'


@viewlet_config(name='map-header', context=Interface, layer=IAdminLayer, view=Interface,
                manager=IMapHeaderViewletManager)
@template_config(template='templates/map-header.pt', layer=IAdminLayer)
class LocationMapHeaderViewlet(Viewlet):
    """Location map header viewlet"""
