#
# Copyright (c) 2008-2017 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from elasticsearch_dsl import Q
from hypatia.interfaces import ICatalog
from hypatia.query import Any
from persistent import Persistent
from zope.container.contained import Contained
from zope.schema.fieldproperty import FieldProperty

from onf_website.component.location.interfaces import ILocationInfo, IViewLocationSettings, \
    VIEW_LOCATION_SETTINGS_KEY
from onf_website.reference.location.interfaces.region import IRegion
from pyams_content.features.search import SearchFolderQuery
from pyams_content.shared.view.interfaces import IViewQueryParamsExtension, IViewSettings, IViewUserQuery, IWfView
from pyams_content_es.interfaces import IViewQueryEsParamsExtension
from pyams_content_es.shared.view import EsSearchFolderQuery
from pyams_utils.adapter import ContextAdapter, adapter_config, get_annotation_adapter
from pyams_utils.factory import factory_config
from pyams_utils.registry import get_utility


try:
    from osgeo.osr import SpatialReference, CoordinateTransformation
    have_gdal = True
except ImportError:
    have_gdal = False


@factory_config(IViewLocationSettings)
class ViewLocationSettings(Persistent, Contained):
    """View location settings"""

    select_context_gps_location = FieldProperty(
        IViewLocationSettings['select_context_gps_location'])
    gps_location = FieldProperty(IViewLocationSettings['gps_location'])
    gps_distance = FieldProperty(IViewLocationSettings['gps_distance'])
    gps_area = FieldProperty(IViewLocationSettings['gps_area'])
    select_context_forests = FieldProperty(IViewLocationSettings['select_context_forests'])
    forests = FieldProperty(IViewLocationSettings['forests'])
    select_context_cities = FieldProperty(IViewLocationSettings['select_context_cities'])
    cities = FieldProperty(IViewLocationSettings['cities'])
    select_context_departments = FieldProperty(
        IViewLocationSettings['select_context_departments'])
    departments = FieldProperty(IViewLocationSettings['departments'])
    select_context_countries = FieldProperty(IViewLocationSettings['select_context_countries'])
    countries = FieldProperty(IViewLocationSettings['countries'])
    select_context_structures = FieldProperty(IViewLocationSettings['select_context_structures'])
    structures = FieldProperty(IViewLocationSettings['structures'])

    @property
    def is_using_context(self):
        return self.select_context_gps_location or self.select_context_forests or \
               self.select_context_cities or self.select_context_departments or \
               self.select_context_countries or self.select_context_structures

    def get_gps_location(self, context):
        if self.select_context_gps_location:
            location = ILocationInfo(context, None)
            if location is not None:
                coords = location.gps_location.wgs_coordinates
                if coords and coords[0] and coords[1]:
                    return {
                        'filter': 'geo_distance',
                        'location': {
                            "lon": coords[0],
                            "lat": coords[1]
                        },
                        'distance': '{0}km'.format(self.gps_distance)
                    }
        else:
            if self.gps_location:
                coords = self.gps_location.wgs_coordinates
                if coords and coords[0] and coords[1]:
                    return {
                        'filter': 'geo_distance',
                        'location': {
                            "lon": coords[0],
                            "lat": coords[1]
                        },
                        'distance': '{0}km'.format(self.gps_distance)
                    }
            elif self.gps_area:
                coords = self.gps_area.wgs_coordinates
                if coords and coords[0] and coords[1]:
                    return {
                        'filter': 'geo_bounding_box',
                        'area': {
                            "top_left": {
                                "lon": coords[0][0],
                                "lat": coords[1][1]
                            },
                            "bottom_right": {
                                "lon": coords[1][0],
                                "lat": coords[0][1]
                            }
                        }
                    }

    def get_forests(self, context):
        forests = set()
        if self.select_context_forests:
            location = ILocationInfo(context, None)
            if location is not None:
                forests |= set(location.forests_index or ())
        if self.forests:
            forests |= set(self.forests)
        return list(forests)

    def get_cities(self, context):
        cities = set()
        if self.select_context_cities:
            location = ILocationInfo(context, None)
            if location is not None:
                cities |= set(location.cities or ())
        if self.cities:
            cities |= set(self.cities)
        return list(cities)

    def get_departments(self, context):
        departments = set()
        if self.select_context_departments:
            location = ILocationInfo(context, None)
            if location is not None:
                departments |= set(location.get_departments())
        if self.departments:
            for value in self.departments:
                if IRegion.providedBy(value):
                    departments |= set(value.departments or ())
                else:
                    departments.add(value)
        return departments

    def get_departments_index(self, context):
        return [dept.insee_code for dept in self.get_departments(context)]

    def get_countries(self, context):
        countries = set()
        if self.select_context_countries:
            location = ILocationInfo(context, None)
            if location is not None:
                countries |= set(location.countries or ())
        if self.countries:
            countries |= self.countries
        return countries

    def get_countries_index(self, context):
        return [country.insee_code for country in self.get_countries(context)]

    def get_structures(self, context):
        structures = set()
        if self.select_context_structures:
            location = ILocationInfo(context, None)
            if location is not None:
                structures |= set(location.structures or ())
        if self.structures:
            structures |= set(self.structures)
        return list(structures)


@adapter_config(context=IWfView,
                provides=IViewLocationSettings)
@adapter_config(name='location',
                context=IWfView,
                provides=IViewSettings)
def view_location_settings_factory(view):
    """View location settings factory"""
    return get_annotation_adapter(view, VIEW_LOCATION_SETTINGS_KEY, IViewLocationSettings,
                                  name='++view:location++')


@adapter_config(name='location',
                context=IWfView,
                provides=IViewQueryParamsExtension)
class ViewLocationQueryParamsExtension(ContextAdapter):
    """View location query params extension"""

    weight = 50

    def get_params(self, context, request=None):
        catalog = get_utility(ICatalog)
        settings = IViewLocationSettings(self.context)
        # check forests
        forests = settings.get_forests(context)
        if forests:
            yield Any(catalog['forests'], forests)
        elif settings.select_context_forests:
            yield None
        # check cities
        cities = settings.get_cities(context)
        if cities:
            yield Any(catalog['cities'], cities)
        elif settings.select_context_cities:
            yield None
        # check departments
        departments = settings.get_departments_index(context)
        if departments:
            yield Any(catalog['departments'], departments)
        elif settings.select_context_departments:
            yield None
        # check countries
        countries = settings.get_countries_index(context)
        if countries:
            yield Any(catalog['countries'], countries)
        elif settings.select_context_countries:
            yield None
        # structures
        structures = settings.get_structures(context)
        if structures:
            yield Any(catalog['structures'], structures)
        elif settings.select_context_structures:
            yield None


@adapter_config(name='location',
                context=IWfView,
                provides=IViewQueryEsParamsExtension)
class ViewLocationQueryEsParamsExtension(ContextAdapter):
    """View location query params extension for Elasticsearch"""

    weight = 50

    def get_es_params(self, context, request=None):
        settings = IViewLocationSettings(self.context)
        # check GPS location
        gps_filter = None
        location = settings.get_gps_location(context)
        if location:
            filter = location.pop('filter')
            filter_args = None
            if filter == 'geo_distance':
                filter_args = {
                    'location.coords': location['location'],
                    'distance': location['distance']
                }
            elif filter == 'geo_bounding_box':
                filter_args = {
                    'location.coords': location['area']
                }
            if filter_args:
                gps_filter = Q(filter, **filter_args)
        if gps_filter:
            yield gps_filter
        elif settings.select_context_gps_location:
            yield None
        # check forests
        forests = settings.get_forests(context)
        if forests:
            yield Q('terms', **{'location.forests': forests})
        elif settings.select_context_forests:
            yield None
        # check cities
        cities = settings.get_cities(context)
        if cities:
            yield Q('terms', **{'location.cities': cities})
        elif settings.select_context_cities:
            yield None
        # check departments
        departments = settings.get_departments_index(context)
        if departments:
            yield Q('terms', **{'location.departments': departments})
        elif settings.select_context_departments:
            yield None
        # check countries
        countries = settings.get_countries_index(context)
        if countries:
            yield Q('terms', **{'location.countries': countries})
        elif settings.select_context_countries:
            yield None
        # check structures
        structures = settings.get_structures(context)
        if structures:
            yield Q('terms', **{'location.structures': structures})
        elif settings.select_context_structures:
            yield None


@adapter_config(name='forest',
                context=SearchFolderQuery,
                provides=IViewUserQuery)
class SearchFolderForestQuery(ContextAdapter):
    """Search folder forest query"""

    @staticmethod
    def get_user_params(request):
        forests = request.params.getall('forest')
        if forests:
            catalog = get_utility(ICatalog)
            yield Any(catalog['forests'], forests)


@adapter_config(name='forest',
                context=EsSearchFolderQuery,
                provides=IViewUserQuery)
class EsSearchFolderForestQuery(ContextAdapter):
    """Search folder forest query for Elasticsearch"""

    @staticmethod
    def get_user_params(request):
        forests = request.params.getall('forest')
        if forests:
            yield Q('terms', **{'location.forests': forests})


@adapter_config(name='location',
                context=EsSearchFolderQuery,
                provides=IViewUserQuery)
class EsSearchFolderLocationQuery(ContextAdapter):
    """Search folder location query"""

    @staticmethod
    def get_user_params(request):
        locations = request.params.getall('location')
        if locations:
            for location in locations:
               yield Q('term', **{'location.departments': location})
