#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from pyramid.decorator import reify
from pyramid.exceptions import NotFound
from pyramid.view import view_config
from z3c.table.column import GetAttrColumn
from z3c.table.interfaces import IColumn, IValues

from onf_website.component.hearing.interfaces import HEARING_SETTINGS, HEARING_SETTINGS_KEYS, IHearingManagerSettings, \
    IHearingManagerTarget
from onf_website.component.hearing.manager import HearingSetting
from pyams_content.interfaces import MANAGE_TOOL_PERMISSION
from pyams_form.help import FormHelp
from pyams_form.interfaces.form import IFormHelp, IWidgetsSuffixViewletsManager
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.table import AttributeSwitcherColumn, BaseTable, I18nColumn, TableViewlet, VisibilitySwitcherColumn
from pyams_skin.viewlet.menu import MenuItem
from pyams_utils.adapter import ContextRequestViewAdapter, adapter_config
from pyams_utils.url import absolute_url
from pyams_viewlet.viewlet import viewlet_config
from pyams_zmi.form import AdminDialogDisplayForm
from pyams_zmi.interfaces.menu import IPropertiesMenu
from pyams_zmi.layer import IAdminLayer

__docformat__ = 'restructuredtext'

from onf_website import _


@viewlet_config(name='hearing-settings.menu', context=IHearingManagerTarget, layer=IAdminLayer,
                manager=IPropertiesMenu, permission=MANAGE_TOOL_PERMISSION, weight=720)
class HearingManagerSettingsMenu(MenuItem):
    """Hearing manager settings menu"""

    label = _("Hearing settings...")
    icon_class = 'fa-volume-up'
    url = 'hearing-settings.html'
    modal_target = True


@pagelet_config(name='hearing-settings.html', context=IHearingManagerTarget, layer=IPyAMSLayer,
                permission=MANAGE_TOOL_PERMISSION)
class HearingManagerSettingsView(AdminDialogDisplayForm):
    """Hearing manager settings view"""

    legend = _("Set access to hearing settings for this tool")


class HearingManagerSettingsTable(BaseTable):
    """Hearing manager settings table"""

    id = 'hearing_manager_table'
    title = _("Hearing settings")
    hide_toolbar = True
    cssClasses = {'table': 'table table-bordered table-striped table-hover table-tight datatable'}
    sortOn = None

    @reify
    def data_attributes(self):
        attributes = super().data_attributes
        attributes.setdefault('table', {}).update({
            'data-ams-location': absolute_url(self.context, self.request),
            'data-ams-datatable-global-filter': 'false',
            'data-ams-datatable-info': 'false',
            'data-ams-datatable-sort': 'false',
            'data-ams-datatable-sdom': "t",
            'data-ams-unfolded-icon-on': 'fa fa-fw fa-folder-open-o',
            'data-ams-unfolded-icon-off': 'fa fa-fw fa-folder-o'

        })
        attributes.setdefault('tr', {}).update({
            'data-ams-element-name': lambda x, col: x.name
        })
        attributes.setdefault('td', {}).update({
            'data-ams-attribute-switcher': self.get_switcher_target,
            'data-ams-switcher-attribute-name': self.get_switcher_attribute
        })
        return attributes

    @staticmethod
    def get_switcher_target(element, column):
        if column.__name__ == 'show-hide':
            return 'switch-hearing-setting-visibility.json'
        elif column.__name__ == 'fold-unfold':
            return 'switch-hearing-setting-folding.json'

    @staticmethod
    def get_switcher_attribute(element, column):
        if column.__name__ == 'show-hide':
            return 'visible'
        elif column.__name__ == 'fold-unfold':
            return 'unfolded'


@adapter_config(name='name', context=(IHearingManagerTarget, IAdminLayer, HearingManagerSettingsTable),
                provides=IColumn)
class HearingManagerSettingsNameColumn(I18nColumn, GetAttrColumn):
    """Hearing manager settings name column"""

    _header = _("Setting type")
    attrName = 'name'
    weight = 10

    def getValue(self, item):
        value = super(HearingManagerSettingsNameColumn, self).getValue(item)
        translate = self.request.localizer.translate
        return translate(HEARING_SETTINGS[value])


@adapter_config(name='show-hide', context=(IHearingManagerTarget, IAdminLayer, HearingManagerSettingsTable),
                provides=IColumn)
class HearingManagerSettingsVisibleColumn(VisibilitySwitcherColumn):
    """Hearing manager settings visibility column"""

    icon_hint = _("Set access or not to this setting in contents")

    weight = 80


@view_config(name='switch-hearing-setting-visibility.json', context=IHearingManagerTarget, request_type=IPyAMSLayer,
             permission=MANAGE_TOOL_PERMISSION, renderer='json', xhr=True)
def switch_hearing_setting_visibility(request):
    """Set hearing setting visibility"""
    settings = IHearingManagerSettings(request.context)
    object_name = request.params.get('object_name')
    setting = settings.get(object_name)
    if setting is None:
        raise NotFound()
    setting.visible = not setting.visible
    settings[object_name] = setting  # for persistence...
    return {
        'status': 'success',
        'visible': setting.visible
    }


@adapter_config(name='fold-unfold', context=(IHearingManagerTarget, IAdminLayer, HearingManagerSettingsTable),
                provides=IColumn)
class HearingManagerSettingsFoldingColumn(AttributeSwitcherColumn):
    """Hearing manager settings folding column"""

    switch_attribute = 'unfolded'

    on_icon_class = 'fa fa-fw fa-folder-open-o'
    off_icon_class = 'fa fa-fw fa-folder-o'

    icon_hint = _("Pre-unfold or not this setting in contents")

    weight = 90


@view_config(name='switch-hearing-setting-folding.json', context=IHearingManagerTarget, request_type=IPyAMSLayer,
             permission=MANAGE_TOOL_PERMISSION, renderer='json', xhr=True)
def switch_hearing_setting_folding(request):
    """Set hearing setting folding"""
    settings = IHearingManagerSettings(request.context)
    object_name = request.params.get('object_name')
    setting = settings.get(object_name)
    if setting is None:
        raise NotFound()
    setting.unfolded = not setting.unfolded
    settings[object_name] = setting  # for persistence...
    return {
        'status': 'success',
        'unfolded': setting.unfolded
    }


@adapter_config(context=(IHearingManagerTarget, IAdminLayer, HearingManagerSettingsTable), provides=IValues)
class HearingManagerSettingsValuesAdapter(ContextRequestViewAdapter):
    """Hearing manager settings table values adapter"""

    @property
    def values(self):
        settings = IHearingManagerSettings(self.context)
        for key in HEARING_SETTINGS_KEYS:
            value = settings.get(key)
            if value is None:
                value = settings[key] = HearingSetting(key)
            yield value


@viewlet_config(name='hearing-settings-table', view=HearingManagerSettingsView, layer=IAdminLayer,
                manager=IWidgetsSuffixViewletsManager)
class HearingManagerSettingsTableViewlet(TableViewlet):
    """Hearing manager settings table viewlet"""

    table_class = HearingManagerSettingsTable


@adapter_config(context=(IHearingManagerTarget, IAdminLayer, HearingManagerSettingsView), provides=IFormHelp)
class HearingManagerSettingsViewHelp(FormHelp):
    """Hearing settings view form help"""

    message = _("Hearing settings are not required for each type of content.\n\n"
                "This form allows you to define which settings are available for this content type, and if so "
                "which ones are unfolded by default.")
    message_format = 'rest'
