#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from elasticsearch_dsl import Q
from hypatia.interfaces import ICatalog
from hypatia.query import Any, Eq
from persistent import Persistent
from zope.container.contained import Contained
from zope.intid.interfaces import IIntIds
from zope.schema.fieldproperty import FieldProperty

from onf_website.component.hearing.interfaces import IHearingInfo, IViewHearingSettings, VIEW_HEARING_SETTINGS_KEY
from onf_website.reference.location.interfaces.region import IRegion
from onf_website.reference.target.interfaces import ITargetTable
from pyams_catalog.query import or_
from pyams_content.shared.common.interfaces import ISharedSite
from pyams_content.shared.view.interfaces import IViewQueryParamsExtension, IViewSettings, IViewUserQuery, IWfView
from pyams_content_es.interfaces import IViewQueryEsParamsExtension
from pyams_content_es.shared.view import EsSearchFolderQuery
from pyams_utils.adapter import ContextAdapter, adapter_config, get_annotation_adapter
from pyams_utils.factory import factory_config
from pyams_utils.registry import get_utility, query_utility


@factory_config(IViewHearingSettings)
class ViewHearingSettings(Persistent, Contained):
    """View hearing settings"""

    select_context_targets = FieldProperty(IViewHearingSettings['select_context_targets'])
    targets = FieldProperty(IViewHearingSettings['targets'])
    use_context_national_scope = FieldProperty(IViewHearingSettings['use_context_national_scope'])
    national_scope = FieldProperty(IViewHearingSettings['national_scope'])
    select_context_forests = FieldProperty(IViewHearingSettings['select_context_forests'])
    forests = FieldProperty(IViewHearingSettings['forests'])
    select_context_cities = FieldProperty(IViewHearingSettings['select_context_cities'])
    cities = FieldProperty(IViewHearingSettings['cities'])
    select_context_departments = FieldProperty(IViewHearingSettings['select_context_departments'])
    departments = FieldProperty(IViewHearingSettings['departments'])
    select_context_countries = FieldProperty(IViewHearingSettings['select_context_countries'])
    countries = FieldProperty(IViewHearingSettings['countries'])
    add_national_scope = FieldProperty(IViewHearingSettings['add_national_scope'])
    select_context_structures = FieldProperty(IViewHearingSettings['select_context_structures'])
    structures = FieldProperty(IViewHearingSettings['structures'])
    select_context_source = FieldProperty(IViewHearingSettings['select_context_source'])
    source_sites = FieldProperty(IViewHearingSettings['source_sites'])
    select_context_sites = FieldProperty(IViewHearingSettings['select_context_sites'])
    diffusion_sites = FieldProperty(IViewHearingSettings['diffusion_sites'])

    @property
    def is_using_context(self):
        return self.select_context_targets or self.use_context_national_scope or self.select_context_forests or \
               self.select_context_cities or self.select_context_departments or self.select_context_countries or \
               self.select_context_structures or self.select_context_source or self.select_context_sites

    def get_targets(self, context):
        targets = set()
        if self.select_context_targets:
            hearing = IHearingInfo(context, None)
            if hearing is not None:
                targets |= set(hearing.targets or ())
        if self.targets:
            targets |= self.targets
        return targets

    def get_targets_index(self, context):
        intids = get_utility(IIntIds)
        return [intids.register(target) for target in self.get_targets(context)]

    def get_national_scope(self, context):
        if self.use_context_national_scope:
            hearing = IHearingInfo(context, None)
            if hearing is not None:
                if hearing.national_scope:
                    return True
        return self.national_scope

    def get_forests(self, context):
        forests = set()
        if self.select_context_forests:
            hearing = IHearingInfo(context, None)
            if hearing is not None:
                forests |= set(hearing.forests_index or ())
        if self.forests:
            forests |= set(self.forests)
        return list(forests)

    def get_cities(self, context):
        cities = set()
        if self.select_context_cities:
            hearing = IHearingInfo(context, None)
            if hearing is not None:
                cities |= set(hearing.cities or ())
        if self.cities:
            cities |= set(self.cities)
        return list(cities)

    def get_departments(self, context):
        departments = set()
        if self.select_context_departments:
            hearing = IHearingInfo(context, None)
            if hearing is not None:
                departments |= set(hearing.get_departments())
        if self.departments:
            for value in self.departments:
                if IRegion.providedBy(value):
                    departments |= set(value.departments or ())
                else:
                    departments.add(value)
        return departments

    def get_departments_index(self, context):
        return [dept.insee_code for dept in self.get_departments(context)]

    def get_countries(self, context):
        countries = set()
        if self.select_context_countries:
            hearing = IHearingInfo(context, None)
            if hearing is not None:
                countries |= set(hearing.countries or ())
        if self.countries:
            countries |= set(self.countries)
        return countries

    def get_countries_index(self, context):
        return [country.insee_code for country in self.get_countries(context)]

    def get_structures(self, context):
        structures = set()
        if self.select_context_structures:
            hearing = IHearingInfo(context, None)
            if hearing is not None:
                structures |= set(hearing.structures or ())
        if self.structures:
            structures |= set(structures)
        return list(structures)

    def get_source_sites(self, context):
        sites = set()
        if self.select_context_source:
            hearing = IHearingInfo(context, None)
            if hearing is not None:
                sites.add(hearing.source)
        if self.source_sites:
            sites |= self.source_sites
        return sites

    def get_source_sites_index(self, context):
        intids = get_utility(IIntIds)
        sites = [query_utility(ISharedSite, name=site) for site in self.get_source_sites(context)]
        return [intids.register(site) for site in sites if site is not None]

    def get_diffusion_sites(self, context):
        sites = set()
        if self.select_context_sites:
            hearing = IHearingInfo(context, None)
            if hearing is not None:
                sites |= set(hearing.diffusion_sites or ())
        if self.diffusion_sites:
            sites |= self.diffusion_sites
        return sites

    def get_diffusion_sites_index(self, context):
        intids = get_utility(IIntIds)
        sites = [query_utility(ISharedSite, name=site) for site in self.get_diffusion_sites(context)]
        return [intids.register(site) for site in sites if site is not None]


@adapter_config(context=IWfView,
                provides=IViewHearingSettings)
@adapter_config(name='hearing',
                context=IWfView,
                provides=IViewSettings)
def view_hearing_settings_factory(view):
    """View hearing settings factory"""
    return get_annotation_adapter(view, VIEW_HEARING_SETTINGS_KEY, IViewHearingSettings,
                                  name='++view:hearing++')


@adapter_config(name='hearing',
                context=IWfView,
                provides=IViewQueryParamsExtension)
class ViewHearingQueryParamsExtension(ContextAdapter):
    """View hearing query params extension"""

    weight = 50

    def get_params(self, context, request=None):
        catalog = get_utility(ICatalog)
        settings = IViewHearingSettings(self.context)
        # check targets
        targets = settings.get_targets_index(context)
        if targets:
            yield Any(catalog['targets'], targets)
        elif settings.select_context_targets:
            yield None
        # check national scope
        scope = settings.get_national_scope(context)
        if scope:
            yield Eq(catalog['national_scope'], scope)
        # check location params
        if settings.add_national_scope:
            location_params = None
            for param in self.get_location_params(catalog, context, settings):
                if param is not None:
                    location_params = or_(location_params, param)
            yield Eq(catalog['national_scope'], True) | location_params
        else:
            yield from self.get_location_params(catalog, context, settings)
        # check structures
        structures = settings.get_structures(context)
        if structures:
            yield Any(catalog['hearing_structures'], structures)
        elif settings.select_context_structures:
            yield None
        # check source sites
        sites = settings.get_source_sites_index(context)
        if sites:
            yield Any(catalog['source_site'], sites)
        elif settings.select_context_source:
            yield None
        # check diffusion sites
        sites = settings.get_diffusion_sites_index(context)
        if sites:
            yield Any(catalog['diffusion_sites'], sites)
        elif settings.select_context_sites:
            yield None

    @staticmethod
    def get_location_params(catalog, context, settings):
        # check forests
        forests = settings.get_forests(context)
        if forests:
            yield Any(catalog['hearing_forests'], forests)
        elif settings.select_context_forests:
            yield None
        # check cities
        cities = settings.get_cities(context)
        if cities:
            yield Any(catalog['hearing_cities'], cities)
        elif settings.select_context_cities:
            yield None
        # check departments
        departments = settings.get_departments_index(context)
        if departments:
            yield Any(catalog['hearing_departments'], departments)
        elif settings.select_context_departments:
            yield None
        # check countries
        countries = settings.get_countries_index(context)
        if countries:
            yield Any(catalog['hearing_countries'], countries)
        elif settings.select_context_countries:
            yield None


@adapter_config(name='hearing',
                context=IWfView,
                provides=IViewQueryEsParamsExtension)
class ViewHearingQueryEsParamsExtension(ContextAdapter):
    """View hearing query params extension for Elasticsearch"""

    weight = 50

    def get_es_params(self, context, request=None):
        settings = IViewHearingSettings(self.context)
        # check targets
        targets = settings.get_targets_index(context)
        if targets:
            yield Q('terms', **{'hearing.targets': targets})
        elif settings.select_context_targets:
            yield None
        # check national scope
        scope = settings.get_national_scope(context)
        if scope:
            yield Q('term', **{'hearing.national_scope': scope})
        # check location params
        if settings.add_national_scope:
            location_params = [param for param in self.get_location_params(context, settings)
                               if param is not None]
            if location_params:
                yield Q('bool', must=Q('term', **{'hearing.national_scope': True})) | \
                      Q('bool', must=location_params)
        else:
            yield from self.get_location_params(context, settings)
        # check structures
        structures = settings.get_structures(context)
        if structures:
            yield Q('terms', **{'hearing.structures': structures})
        elif settings.select_context_structures:
            yield None
        # check sources
        sources = settings.get_source_sites_index(context)
        if sources:
            yield Q('terms', **{'hearing.source_site': sources})
        elif settings.select_context_source:
            yield None
        # check diffusion sites
        sites = settings.get_diffusion_sites_index(context)
        if sites:
            yield Q('terms', **{'hearing.diffusion_sites': sites})
        elif settings.select_context_sites:
            yield None

    @staticmethod
    def get_location_params(context, settings):
        forests = settings.get_forests(context)
        if forests:
            yield Q('terms', **{'hearing.forests': forests})
        elif settings.select_context_forests:
            yield None
        # check cities
        cities = settings.get_cities(context)
        if cities:
            yield Q('terms', **{'hearing.cities': cities})
        elif settings.select_context_cities:
            yield None
        # check departments
        departments = settings.get_departments_index(context)
        if departments:
            yield Q('terms', **{'hearing.departments': departments})
        elif settings.select_context_departments:
            yield None
        # check countries
        countries = settings.get_countries_index(context)
        if countries:
            yield Q('terms', **{'hearing.countries': countries})
        elif settings.select_context_countries:
            yield None


@adapter_config(name='hearing',
                context=EsSearchFolderQuery,
                provides=IViewUserQuery)
class EsSearchFolderHearingQuery(ContextAdapter):
    """Search folder hearing query for Elasticsearch"""

    @staticmethod
    def get_user_params(request):
        hearings = request.params.getall('hearing')
        if hearings:
            target_table = query_utility(ITargetTable)
            results = []
            terms = [v for v in target_table.values()]
            for term_name in hearings:
                term_object = next((
                    term
                    for term in terms
                    if (getattr(term, 'facets_label', None) and term.facets_label.get('fr', '') == term_name) or
                       (not getattr(term, 'facets_label', None) and term.title.get('fr', '') == term_name)
                ), None)
                if term_object is not None:
                    intids = query_utility(IIntIds)
                    results.append(intids.queryId(term_object))
                    yield Q('term', **{'hearing.targets': intids.queryId(term_object)})
