#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from persistent import Persistent
from zope.location import Location
from zope.schema.fieldproperty import FieldProperty

from onf_website.component.gallery.interfaces import GALLERY_PARAGRAPH_BEFOREAFTER_SETTINGS_KEY, \
    GALLERY_PARAGRAPH_MOSAIC_SETTINGS_KEY, IGalleryParagraphBeforeAfterRendererSettings, \
    IGalleryParagraphMosaicRendererSettings
from onf_website.skin.public import baguettebox, jquery_twentytwenty_init, magic_grid, \
    owl_carousel_init
from pyams_content.component.gallery.interfaces import IBaseGallery
from pyams_content.features.renderer.interfaces import IContentRenderer
from pyams_default_theme.features.renderer import BaseContentRenderer
from pyams_skin.layer import IPyAMSLayer
from pyams_template.template import template_config
from pyams_utils.adapter import adapter_config, get_annotation_adapter
from pyams_utils.factory import factory_config


__docformat__ = 'restructuredtext'

from onf_website import _


class BaseGalleryRenderer(BaseContentRenderer):
    """Base gallery renderer"""

    i18n_context_attrs = ('title',)


@adapter_config(name='default',
                context=(IBaseGallery, IPyAMSLayer),
                provides=IContentRenderer)
@template_config(template='templates/gallery-default.pt',
                 layer=IPyAMSLayer)
@template_config(name='PARA:title-left',
                 template='templates/gallery-title-left.pt',
                 layer=IPyAMSLayer)
class ONFDefaultGalleryCarouselParagraphRenderer(BaseGalleryRenderer):
    """Default gallery renderer"""

    label = _("ONF: Full size carousel")
    weight = 1

    resources = (owl_carousel_init,)


@adapter_config(name='carousel-sm',
                context=(IBaseGallery, IPyAMSLayer),
                provides=IContentRenderer)
@template_config(template='templates/gallery-carousel.pt',
                 layer=IPyAMSLayer)
@template_config(name='PARA:title-left',
                 template='templates/gallery-title-left.pt',
                 layer=IPyAMSLayer)
class ONFGalleryCarouselParagraphRenderer(BaseGalleryRenderer):
    """Small gallery renderer"""

    label = _("ONF: Horizontal carousel")
    weight = 2

    resources = (owl_carousel_init,)


#
# Before/after gallery renderer
#

@factory_config(IGalleryParagraphBeforeAfterRendererSettings)
class GalleryParagraphBeforeAfterRendererSettings(Persistent, Location):
    """Gallery paragraph before/after renderer settings"""

    description = FieldProperty(IGalleryParagraphBeforeAfterRendererSettings['description'])


@adapter_config(context=IBaseGallery,
                provides=IGalleryParagraphBeforeAfterRendererSettings)
def gallery_paragraph_beforeafter_renderer_settings_factory(context):
    """Gallery paragraph before/after renderer settings factory"""
    return get_annotation_adapter(context, GALLERY_PARAGRAPH_BEFOREAFTER_SETTINGS_KEY,
                                  IGalleryParagraphBeforeAfterRendererSettings)


@adapter_config(name='before-after',
                context=(IBaseGallery, IPyAMSLayer),
                provides=IContentRenderer)
@template_config(template='templates/gallery-before-after.pt',
                 layer=IPyAMSLayer)
@template_config(name='PARA:title-left',
                 template='templates/gallery-title-left.pt',
                 layer=IPyAMSLayer)
class ONFGalleryParagraphBeforeAfterRenderer(BaseGalleryRenderer):
    """ONF gallery before/after renderer"""

    label = _("ONF: Dual images comparison")
    weight = 3

    image_before = None
    image_after = None

    def update(self):
        super(ONFGalleryParagraphBeforeAfterRenderer, self).update()
        medias = self.context.get_visible_medias()
        self.image_before = next(medias, None)
        self.image_after = next(medias, None)

    resources = (jquery_twentytwenty_init,)
    settings_interface = IGalleryParagraphBeforeAfterRendererSettings


#
# Mosaic gallery renderer
#

@factory_config(IGalleryParagraphMosaicRendererSettings)
class GalleryParagraphMosaicRendererSettings(Persistent, Location):
    """Gallery paragraph mosaic renderer settings"""

    thumbnails_count = FieldProperty(IGalleryParagraphMosaicRendererSettings['thumbnails_count'])
    button_label = FieldProperty(IGalleryParagraphMosaicRendererSettings['button_label'])

    def __init__(self):
        super().__init__()
        self.button_label = {
            'en': "Display all images",
            'fr': "Afficher toutes les images"
        }


@adapter_config(context=IBaseGallery,
                provides=IGalleryParagraphMosaicRendererSettings)
def gallery_paragraph_mosaic_renderer_settings_factory(context):
    """Gallery paragraph mosaic renderer settings factory"""
    return get_annotation_adapter(context, GALLERY_PARAGRAPH_MOSAIC_SETTINGS_KEY,
                                  IGalleryParagraphMosaicRendererSettings)


@adapter_config(name='mosaic',
                context=(IBaseGallery, IPyAMSLayer),
                provides=IContentRenderer)
@template_config(template='templates/gallery-mosaic.pt',
                 layer=IPyAMSLayer)
class ONFGalleryMosaicRenderer(BaseGalleryRenderer):
    """ONF gallery mosaic renderer"""

    label = _("ONF: Images grid")
    weight = 4

    resources = (baguettebox, )
    settings_interface = IGalleryParagraphMosaicRendererSettings
